cimport atFunctions.catFunctions as at_
from libc.math cimport cos, sin, sqrt, atan2
import numpy as np
from functools import reduce
import math

# "cimport" is used to import special compile-time information
# about the numpy module (this is stored in a file numpy.pxd which is
# currently part of the Cython distribution).
cimport numpy as np


##############################
# atFunctions constant values
##############################

PI = 3.1415926535897932385
TWO_PI = 6.283185307179586477
DEG2RAD = 0.017453292519943295769
RAD2DEG = 57.295779513082320877

# GRS80, http://dgfi2.dgfi.badw-muenchen.de/geodis/REFS/grs80.html
EARTH_RADIUS = 6378.137
EARTH_E2 = 0.00669438002290

MOON_RADIUS = 1738.
AU = 149597870.
EPS = 1.e-12
MJD_B1950 = 33281.923
MJD_J2000 = 51544.500

SAA_NOMINAL = 0
SAA_ASCA_SIS = 1
SAA_ASCA_STT = 2
SAA_SUZAKU_HXD = 3

# atFunctions-numpy structure mapping

AtEulerAng = [
  ('phi', 'f8'),
  ('theta', 'f8'),
  ('psi', 'f8'),
]

AtPolarVect = [
  ('r', 'f8'),
  ('lon', 'f8'),
  ('lat', 'f8'),
]
AtRightAscension = [
  ('hour', 'i4'),
  ('min', 'i4'),
  ('sec', 'f8'),
]
AtDeclination = [
  ('sign', 'i4'),
  ('deg', 'u4'),
  ('min', 'u4'),
  ('sec', 'f8'),
]
AtPolarVect60 = [
  ('ra', AtRightAscension),
  ('dec', AtDeclination),
]
AtTime = [
  ('yr', 'i4'),
  ('mo', 'i4'),
  ('dy', 'i4'),
  ('hr', 'i4'),
  ('mn', 'i4'),
  ('sc', 'i4'),
  ('ms', 'f4'),
]
AtTimeD = [
  ('yr', 'i4'),
  ('mo', 'i4'),
  ('dy', 'i4'),
  ('hr', 'i4'),
  ('mn', 'i4'),
  ('sc', 'i4'),
  ('ss', 'f8'),
]
AtElement = [
  ('itz', AtTime),
  ('semiax', 'f8'),
  ('eccent', 'f8'),
  ('aincln', 'f8'),
  ('ragome', 'f8'),
  ('smaome', 'f8'),
  ('omean0', 'f8'),
  ('adot', 'f8'),
  ('eccdot', 'f8'),
  ('aindot', 'f8'),
  ('ragdot', 'f8'),
  ('smodot', 'f8'),
  ('znbar', 'f8'),
  ('znbadt', 'f8'),
  ('mjdz', 'f8'),
  ('perige', 'f8'),
  ('apoge', 'f8'),
]
AtElement3 = [
  ('itz', AtTimeD),
  ('itn', AtTimeD),
  ('mjdz', 'f8'),
  ('mjdn', 'f8'),
  ('semiax', 'f8'),
  ('eccent', 'f8'),
  ('aincln', 'f8'),
  ('ragome', 'f8'),
  ('smaome', 'f8'),
  ('omean0', 'f8'),
  ('adot', 'f8'),
  ('eccdot', 'f8'),
  ('aindot', 'f8'),
  ('ragdot', 'f8'),
  ('smodot', 'f8'),
  ('znbar', 'f8'),
  ('znbadt', 'f8'),
  ('perige', 'f8'),
  ('apoge', 'f8'),
]

def n2np(x, dtype=np.double):
  if isinstance(x, np.ndarray):
    return x
  return np.array([float(x)], dtype=dtype)

def array_mul(x):
  def _mul(y, z):
    return y * z
  return reduce(_mul, x)

def is_number(x):
  try:
    if isinstance(x, np.ndarray) and x.ndim != 0:
      return False
    _ = float(x)
    return True
  except ValueError:
    return False

def valid_array_len(*x):
  y = [a.shape[0] for a in x if 1 < a.shape[0]]
  return 0 == len(y) or max(y) == min(y)

# ndarray to atFunctions struct utilities
cdef int toVect(np.ndarray s, at_.AtVect out) except -1:
  if s.ndim != 1:
    raise ValueError("Invalid dimention s")
  for i in range(3):
    out[i] = s[i]
  return 1

cdef int fromVect(at_.AtVect s, np.ndarray out) except -1:
  if out.ndim != 1:
    raise ValueError("Invalid dimention out")
  for i in range(3):
    out[i] = s[i]
  return 1

cdef int toRotMat(np.ndarray s, at_.AtRotMat out) except -1:
  if s.ndim != 2:
    raise ValueError("Invalid dimention s")
  for i in range(3):
    for j in range(3):
      out[i][j] = s[i][j]
  return 1

cdef int fromRotMat(at_.AtRotMat s, np.ndarray out) except -1:
  if out.ndim != 2:
    raise ValueError("Invalid dimention: out")
  for i in range(3):
    for j in range(3):
      out[i][j] = s[i][j]
  return 1

cdef int toEulerAng(s, at_.AtEulerAng *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.phi = s['phi']
  out.theta = s['theta']
  out.psi = s['psi']
  return 1

cdef int fromEulerAng(at_.AtEulerAng *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out['phi'] = s.phi
  out['theta'] = s.theta
  out['psi'] = s.psi
  return 1

cdef int toQuat(np.ndarray s, at_.AtQuat out) except -1:
  if s.ndim != 1:
    raise ValueError("Invalid dimention: s")
  for i in range(4):
    out[i] = s[i]
  return 1

cdef int fromQuat(at_.AtQuat s, np.ndarray out) except -1:
  if out.ndim != 1:
    raise ValueError("Invalid dimention: out")
  for i in range(4):
    out[i] = s[i]
  return 1

cdef int toPolarVect(s, at_.AtPolarVect *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.r   = s['r']
  out.lon = s['lon']
  out.lat = s['lat']
  return 1

cdef int fromPolarVect(at_.AtPolarVect *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out['r']   = s.r
  out['lon'] = s.lon
  out['lat'] = s.lat
  return 1

cdef int toRA(s, at_.AtRightAscension *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.hour = s['hour']
  out.min  = s['min']
  out.sec  = s['sec']
  return 1

cdef int fromRA(at_.AtRightAscension *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out[0] = s.hour
  out[1] = s.min
  out[2] = s.sec
  return 1

cdef int toDec(s, at_.AtDeclination *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.sign = s['sign']
  out.deg = s['deg']
  out.min = s['min']
  out.sec = s['sec']
  return 1

cdef int fromDec(at_.AtDeclination *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out['sign'] = s.sign
  out['deg'] = s.deg
  out['min'] = s.min
  out['sec'] = s.sec
  return 1

cdef int toPolarVect60(s, at_.AtPolarVect60 *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  toRA(s['ra'], &out.ra)
  toDec(s['dec'], &out.dec)
  return 1

cdef int fromPolarVect60(at_.AtPolarVect60 *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  fromRA(&s.ra, out['ra'])
  fromDec(&s.dec, out['dec'])
  return 1

cdef int toAtTime(s, at_.AtTime *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.yr = s['yr']
  out.mo = s['mo']
  out.dy = s['dy']
  out.hr = s['hr']
  out.mn = s['mn']
  out.sc = s['sc']
  out.ms = s['ms']
  return 1

cdef int fromAtTime(at_.AtTime *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out['yr'] = s.yr
  out['mo'] = s.mo
  out['dy'] = s.dy
  out['hr'] = s.hr
  out['mn'] = s.mn
  out['sc'] = s.sc
  out['ms'] = s.ms
  return 1

cdef int toAtTimeD(s, at_.AtTimeD *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  out.yr = s['yr']
  out.mo = s['mo']
  out.dy = s['dy']
  out.hr = s['hr']
  out.mn = s['mn']
  out.sc = s['sc']
  out.ss = s['ss']
  return 1

cdef int fromAtTimeD(at_.AtTimeD *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  out['yr'] = s.yr
  out['mo'] = s.mo
  out['dy'] = s.dy
  out['hr'] = s.hr
  out['mn'] = s.mn
  out['sc'] = s.sc
  out['ss'] = s.ss
  return 1

cdef int toAtElement3(s, at_.AtElement3 *out) except -1:
  if s.ndim != 0:
    raise ValueError("Invalid dimention: s")
  toAtTimeD(s['itz'], &out.itz)
  toAtTimeD(s['itn'], &out.itn)
  out.mjdz = s['mjdz']
  out.mjdn = s['mjdn']
  out.semiax = s['semiax']
  out.eccent = s['eccent']
  out.aincln = s['aincln']
  out.ragome = s['ragome']
  out.smaome = s['smaome']
  out.omean0 = s['omean0']
  out.adot = s['adot']
  out.eccdot = s['eccdot']
  out.aindot = s['aindot']
  out.ragdot = s['ragdot']
  out.smodot = s['smodot']
  out.znbar = s['znbar']
  out.znbadt = s['znbadt']
  out.perige = s['perige']
  out.apoge = s['apoge']
  return 1

cdef int fromAtElement3(at_.AtElement3 *s, out) except -1:
  if out.ndim != 0:
    raise ValueError("Invalid dimention: out")
  fromAtTimeD(&s.itz, out['itz'])
  fromAtTimeD(&s.itn, out['itn'])
  out['semiax'] = s.semiax
  out['eccent'] = s.eccent
  out['aincln'] = s.aincln
  out['ragome'] = s.ragome
  out['smaome'] = s.smaome
  out['omean0'] = s.omean0
  out['adot'] = s.adot
  out['eccdot'] = s.eccdot
  out['aindot'] = s.aindot
  out['ragdot'] = s.ragdot
  out['smodot'] = s.smodot
  out['znbar'] = s.znbar
  out['znbadt'] = s.znbadt
  out['perige'] = s.perige
  out['apoge'] = s.apoge
  return 1



def atRAToRadian(np.ndarray ra):
  """\
  covert Right Ascension in hr,min,s to radian.

  ndarray[1] atRaToRadian(ra[3])
  ndarray[N,1] atRaToRadian(ra[N,3])
  """
  cdef at_.AtRightAscension at_ra
  expand = False
  if 0 == ra.ndim:
    expand = True
    ra = ra.reshape(1)
  if 1 != ra.ndim:
    raise ValueError("Invalid dimention: ra")
  cdef unsigned int n, i
  n = ra.shape[0]
  cdef np.ndarray r = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    toRA(ra[i], &at_ra)
    r[i] = at_.atRAToRadian(at_ra)
  if expand: return r[0]
  return r

def atDecToRadian(np.ndarray dec):
  """
  covert Declination in deg,min,s to radian.

  ndarray[1] atRaToRadian(dec[3])
  ndarray[N,1] atRaToRadian(dec[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtDeclination at_dec
  expand = False
  if 0 == dec.ndim:
    expand = True
    dec = dec.reshape(1)
  if 1 != dec.ndim:
    raise ValueError("Invalid dimention: dec")
  n = dec.shape[0]
  cdef np.ndarray d = np.zeros((n, ), dtype=np.double)
  for i in range(dec.shape[0]):
    toDec(dec[i], &at_dec)
    d[i] = at_.atDecToRadian(at_dec)
  if expand: return d[0]
  return d

def atPol60ToVect(np.ndarray p):
  """
  covert R.A. and Dec in hr/deg,min,s to Vector.
  ndarray[3] atPol60ToVect(p[7])
  ndarray[N,3] atPol60ToVect(p[N,7])
  """
  cdef unsigned int n, i
  cdef np.ndarray v
  cdef at_.AtPolarVect60 at_p
  cdef at_.AtVect at_v
  expand = False
  if p.ndim == 0:
    expand = True
    p = p.reshape(1)
  if p.ndim != 1:
    raise ValueError("Invalid dimention: p")
  n = p.shape[0]
  v = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toPolarVect60(p[i], &at_p)
    at_.atPol60ToVect(&at_p, at_v)
    fromVect(at_v, v[i])
  if expand: return v[0]
  return v

def atVectToPol60(np.ndarray x):
  """
  covert Vector to R.A. and Dec in hr/deg,min,s.
  ndarray[7] atVectorToPol60(ndarray[3])
  ndarray[N,7] atVectorToPol60(ndarray[N,3])
  """
  cdef unsigned int n, i
  expand = False
  if x.ndim == 1:
    expand = True
    x = x.reshape(1, 3)
  if x.ndim != 2:
    raise ValueError("Invalid dimention")
  n = x.shape[0]
  cdef at_.AtVect at_x
  cdef at_.AtPolarVect60 at_p
  cdef np.ndarray out = np.zeros((n, ), dtype=AtPolarVect60)
  for i in range(n):
    toVect(x[i], at_x)
    at_.atVectToPol60(at_x, &at_p)
    fromPolarVect60(&at_p, out[i])
  if expand: return out[0]
  return out

def atVectToPolDeg(np.ndarray x):
  """
  covert Vector to R.A. and Dec in degree.

  ndarray<('r','alpha','delta')> atVectToPoDeg(AtVect)
  ndarray<('r','alpha','delta')>[] atVectToPoDeg(AtVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x
  cdef double c_r, c_alpha, c_delta
  expand = False
  if x.ndim == 1:
    expand = True
    x = x.reshape(1, 3)
  if x.ndim != 2:
    raise ValueError("Invalid dimention")
  n = x.shape[0]
  cdef np.ndarray r = np.zeros((n, ), dtype='f8')
  cdef np.ndarray alpha = np.zeros((n, ), dtype='f8')
  cdef np.ndarray delta = np.zeros((n, ), dtype='f8')
  for i in range(n):
    toVect(x[i], at_x)
    at_.atVectToPolDeg(at_x, &c_r, &c_alpha, &c_delta)
    r[i] = c_r
    alpha[i] = c_alpha
    delta[i] = c_delta
  if expand: return r[0], alpha[0], delta[0]
  return r, alpha, delta

def atPolDegToVect(r, alpha, delta):
  """\
  covert R.A. and Dec in degree to Vector

  AtVect atPolDegToVect(double, double, double)
   :
  AtVect[] atPolDegToVect(double[], double[], double[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x
  expand = False
  if is_number(r) and is_number(alpha) and is_number(delta):
    expand = True
  r = n2np(r)
  alpha = n2np(alpha)
  delta = n2np(delta)
  if r.ndim != 1:
    raise ValueError("Invalid dimention: r")
  if alpha.ndim != 1:
    raise ValueError("Invalid dimention: alpha")
  if delta.ndim != 1:
    raise ValueError("Invalid dimention: delta")
  if not valid_array_len(r, alpha, delta):
    raise ValueError("Invalid array length")
  n = max(r.shape[0], alpha.shape[0], delta.shape[0])
  cdef np.ndarray x = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    at_.atPolDegToVect(
      r[min(i, r.shape[0] - 1)],
      alpha[min(i, alpha.shape[0] - 1)],
      delta[min(i, delta.shape[0] - 1)],
      at_x
    )
    fromVect(at_x, x[i])
  if expand: return x[0]
  return x

def atRotVect(np.ndarray rm, np.ndarray x):
  """\
  ROTATE A VECTOR WITH ROTATION MATRIX.

  ndarray[3] = atRotVect(np.ndarray[3,3], np.ndarray[3])
  ndarray[N,3] = atRotVect(np.ndarray[N,3,3], np.ndarray[3])
  ndarray[N,3] = atRotVect(np.ndarray[3,3], np.ndarray[N,3])
  ndarray[N,3] = atRotVect(np.ndarray[N,3,3], np.ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtRotMat at_rm
  cdef at_.AtVect at_x, at_y
  expand = False
  if rm.ndim == 2 and x.ndim == 1:
    expand = True
  if rm.ndim == 2:
    rm = rm.reshape(1, 3, 3)
  if x.ndim == 1:
    x = x.reshape(1, 3)
  if rm.ndim != 3:
    raise ValueError("Invalid dimention rm")
  if x.ndim != 2:
    raise ValueError("Invalid dimention x")
  if not valid_array_len(rm, x):
    raise ValueError("Invalid array length")
  n = max(rm.shape[0], x.shape[0])
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toRotMat(rm[min(i, rm.shape[0] - 1)], at_rm)
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    at_.atRotVect(at_rm, at_x, at_y)
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y

def atSetRotMat(np.ndarray axis, roll):
  """\
  CALC ROTATION MATRIX FOR AXIS AND ROLL ANGLE.

  ndarray[3,3] atSetRotMat(ndarray[3], double)
  ndarray[N,3,3] atSetRotMat(ndarray[N,3], double)
  ndarray[N,3,3] atSetRotMat(ndarray[3], ndarray[N])
  ndarray[N,3,3] atSetRotMat(ndarray[N,3], ndarray[N])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_axis
  cdef at_.AtRotMat at_rm
  expand = False
  if axis.ndim == 1 and is_number(roll):
    expand = True
  if axis.ndim == 1:
    axis = axis.reshape((1, 3))
  roll = n2np(roll)
  if axis.ndim != 2:
    raise ValueError("Invalid dimention")
  if roll.ndim != 1:
    raise ValueError("Invalid dimention")
  if not valid_array_len(axis, roll):
    raise ValueError("Invalid array length")
  n = max(axis.shape[0], roll.shape[0])
  cdef np.ndarray rm = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    toVect(axis[min(i, axis.shape[0] - 1)], at_axis)
    at_.atSetRotMat(at_axis, roll[min(i, roll.shape[0] - 1)], at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm

def atSetRotMatZX(np.ndarray zAxis, np.ndarray xAxis):
  """\
  ROTATION MATRIX defined by New Z-axis and a vector in (+)X-Z half plane.

  ndarray[3,3] atSetRotMatZX(ndarray[3], ndarray[3])
  ndarray[N,3,3] atSetRotMatZX(ndarray[N,3], ndarray[3])
  ndarray[N,3,3] atSetRotMatZX(ndarray[N,3], ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_zAxis, at_xAxis
  cdef at_.AtRotMat at_rm
  expand = False
  if zAxis.ndim == 1 and xAxis.ndim == 1:
    expand = True
  if zAxis.ndim == 1:
    zAxis = zAxis.reshape((1, 3))
  if xAxis.ndim == 1:
    xAxis = xAxis.reshape((1, 3))
  if zAxis.ndim != 2:
    raise ValueError("Invalid dimention zAxis")
  if xAxis.ndim != 2:
    raise ValueError("Invalid dimention xAxis")
  if not valid_array_len(zAxis, xAxis):
    raise ValueError("Invalid array length")
  n = max(zAxis.shape[0], xAxis.shape[0])
  cdef np.ndarray rm = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    toVect(zAxis[min(i, zAxis.shape[0] - 1)], at_zAxis)
    toVect(xAxis[min(i, xAxis.shape[0] - 1)], at_xAxis)
    at_.atSetRotMatZX(at_zAxis, at_xAxis, at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm


def atInvRotMat(np.ndarray rm):
  """\
  CALC INVERSE ROTATION MATRIX

  ndarray[3,3] atInvRotMat(ndarray[3,3])
  ndarray[N,3,3] atInvRotMat(ndarray[N,3,3])
  """
  cdef unsigned int n, i
  cdef at_.AtRotMat at_rm
  cdef at_.AtRotMat at_rm2
  expand = False
  if rm.ndim == 2:
    expand = True
    rm = rm.reshape((1, 3, 3))
  if rm.ndim != 3:
    raise ValueError("Invalid dimention: rm")
  n = rm.shape[0]
  cdef np.ndarray rm2 = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    toRotMat(rm[i], at_rm)
    at_.atInvRotMat(at_rm, at_rm2)
    fromRotMat(at_rm2, rm2[i])
  if expand: return rm2[0]
  return rm2


def atNormVect(x):
  """\
  NORMALIZE A VECTOR.
  ndarray[3] atNormVect(ndarray[3])
  ndarray[N,3] atNormVect(ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x, at_y
  expand = False
  if x.ndim == 1:
    expand = True
    x = x.reshape((1, 3))
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  n = x.shape[0]
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toVect(x[i], at_x)
    at_.atNormVect(at_x, at_y)
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y


def atNorm(np.ndarray x):
  """\
  Norm (Absolute size) of A VECTOR.

  double atNorm(np.ndarray[3])
  ndarray[N] atNorm(np.ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x
  expand = False
  if x.ndim == 1:
    expand = True
    x = x.reshape((1, 3))
  if x.ndim != 2:
    raise ValueError("Invlid dimention: x")
  n = x.shape[0]
  cdef np.ndarray norm = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    toVect(x[i], at_x)
    norm[i] = at_.atNorm(at_x)
  if expand: return norm[0]
  return norm


def atRMProd(np.ndarray rm0, np.ndarray rm1):
  """\
  product of two rotation matrices rm2 = rm1 rm0

  ndarray[3,3] = atRMProd(ndarray[3,3], ndarray[3,3])
  ndarray[N,3,3] = atRMProd(ndarray[N,3,3], ndarray[3,3])
  ndarray[N,3,3] = atRMProd(ndarray[3,3], ndarray[N,3,3])
  ndarray[N,3,3] = atRMProd(ndarray[N,3,3], ndarray[N,3,3])
  """
  expand = False
  if rm0.ndim == 2 and rm1.ndim == 2:
    expand = True
  if rm0.ndim == 2:
    rm0 = rm0.reshape((1, 3, 3))
  if rm1.ndim == 2:
    rm1 = rm1.reshape((1, 3, 3))
  if rm0.ndim != 3:
    raise ValueError("Invalid dimention: rm0")
  if rm1.ndim != 3:
    raise ValueError("Invalid dimention: rm1")
  if not valid_array_len(rm0, rm1):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(rm0.shape[0], rm1.shape[0])
  cdef np.ndarray rm2 = np.zeros((n, 3, 3), dtype=np.double)
  cdef at_.AtRotMat at_rm0
  cdef at_.AtRotMat at_rm1
  cdef at_.AtRotMat at_rm2
  for i in range(n):
    toRotMat(rm0[min(i, rm0.shape[0] - 1)], at_rm0)
    toRotMat(rm1[min(i, rm1.shape[0] - 1)], at_rm1)
    at_.atRMProd(at_rm0, at_rm1, at_rm2)
    fromRotMat(at_rm2, rm2[i])
  if expand: return rm2[0]
  return rm2


def atRMCheck(np.ndarray rm):
  """\
  Checking consistency (unitarity) of a rotation matrix

  note that C-atFunctions read/write variable but python return new value
  ndarray[3,3] atRMCheck(ndarray[3,3])
  ndarray[N,3,3] atRMCheck(ndarray[N,3,3])
  """
  expand = False
  if rm.ndim == 2:
    expand = True
    rm = rm.reshape((1, 3, 3))
  if rm.ndim != 3:
    raise ValueError("Invalid dimention: rm")
  cdef unsigned int n, i
  n = rm.shape[0]
  cdef np.ndarray rm2 = np.ndarray((n, 3, 3), dtype=np.double)
  cdef at_.AtRotMat at_rm
  for i in range(n):
    toRotMat(rm[i], at_rm)
    at_.atRMCheck(at_rm)
    fromRotMat(at_rm, rm2[i])
  if expand: return rm2[0]
  return rm2


def atRMToEuler(np.ndarray rm):
  """\
  FIND EULER ANGLES FOR A ROTATION MATRI

  ndarray[3] atRMToEuler(ndarray[3,3])
  ndarray[N,3] atRMToEuler(ndarray[N,3,3])
  """
  expand = False
  if rm.ndim == 2:
    expand = True
    rm = rm.reshape((1, 3, 3))
  if rm.ndim != 3:
    raise ValueError("Invalid dimention: rm")
  cdef unsigned int n, i
  n = rm.shape[0]
  cdef np.ndarray ea = np.ndarray((n, ), dtype=AtEulerAng)
  cdef at_.AtEulerAng at_ea
  cdef at_.AtRotMat at_rm
  for i in range(n):
    toRotMat(rm[i], at_rm)
    at_.atRMToEuler(at_rm, &at_ea)
    fromEulerAng(&at_ea, ea[i])
  if expand: return ea[0]
  return ea


def atEulerToRM(np.ndarray ea):
  """\
  FIND ROTATION MATRIX FOR A EULER ANGLE SET.

  ndarray[3,3] atEulerToRM(ndarray[3])
  ndarray[N,3,3] atEulerToRM(ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtEulerAng at_ea
  cdef at_.AtRotMat at_rm
  expand = False
  if ea.ndim == 0:
    expand = True
    ea = ea.reshape(1)
  if ea.ndim != 1:
    raise ValueError("Invalid dimention: ea")
  n = ea.shape[0]
  cdef np.ndarray rm = np.ndarray((n, 3, 3), dtype=np.double)
  for i in range(n):
    toEulerAng(ea[i], &at_ea)
    at_.atEulerToRM(&at_ea, at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm


def atRMToQuat(np.ndarray rm):
  """\
  convert a Rotation Matrix to Quaternion.

  ndarray[4] atRMToQuat(ndarray[3,3])
  ndarray[N,4] atRMToQuat(ndarray[N,3,3])
  """
  expand = False
  if rm.ndim == 2:
    expand = True
    rm = rm.reshape((1, 3, 3))
  if rm.ndim != 3:
    raise ValueError("Invalid dimention: rm")
  cdef unsigned int n, i
  cdef at_.AtRotMat at_rm
  cdef at_.AtQuat at_q
  cdef np.ndarray q
  n = rm.shape[0]
  q = np.ndarray((n, 4), dtype=np.double)
  for i in range(n):
    toRotMat(rm[i], at_rm)
    at_.atRMToQuat(at_rm, at_q)
    fromQuat(at_q, q[i])
  if expand: return q[0]
  return q


def atQuatToRM(np.ndarray q):
  """\
  FIND ROTATION MATRIX FOR A EULER ANGLE SET.

  ndarray[3,3] atQuatToRM(ndarray[4])
  ndarray[N,3,3] atQuatToRM(ndarray[N,4])
  """
  expand = False
  if q.ndim == 1:
    expand = True
    q = q.reshape((1, 4))
  if q.ndim != 2:
    raise ValueError("Invalid dimention: q")
  cdef at_.AtQuat at_q
  cdef at_.AtRotMat at_rm
  cdef np.ndarray rm
  cdef unsigned int n, i
  n = q.shape[0]
  rm = np.ndarray((n, 3, 3), dtype=np.double)
  for i in range(n):
    toQuat(q[i], at_q)
    at_.atQuatToRM(at_q, at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm


def atQuatProd(np.ndarray q0, np.ndarray q1):
  """\
  product of two quaternion: q2 = q0 q1
  (in matrix representation: rm(q2) = rm(q1) rm(q0).
  note the inverse order of quaternion as compared to matrix )

  ndarray[4] atQuatProd(ndarray[4], ndarray[4])
  ndarray[N,4] atQuatProd(ndarray[N,4], ndarray[4])
  ndarray[N,4] atQuatProd(ndarray[4], ndarray[N,4])
  ndarray[N,4] atQuatProd(ndarray[N,4], ndarray[N,4])
  """
  expand = False
  if q0.ndim == 1 and q1.ndim == 1:
    expand = True
  if q0.ndim == 1:
    q0 = q0.reshape((1, 4))
  if q1.ndim == 1:
    q1 = q1.reshape(1, 4)
  if q0.ndim != 2:
    raise ValueError("Invalid dimention: q0")
  if q1.ndim != 2:
    raise ValueError("Invalid dimention: q1")
  if not valid_array_len(q0, q1):
    raise ValueError("Invalid array length")
  cdef at_.AtQuat at_q0
  cdef at_.AtQuat at_q1
  cdef at_.AtQuat at_q2
  cdef np.ndarray q2
  cdef unsigned int n, i
  n = max(q0.shape[0], q1.shape[0])
  q2 = np.zeros((n, 4), dtype=np.double)
  for i in range(n):
    toQuat(q0[min(i, q0.shape[0] - 1)].copy(), at_q0)
    toQuat(q1[min(i, q1.shape[0] - 1)].copy(), at_q1)
    at_.atQuatProd(at_q0, at_q1, at_q2)
    fromQuat(at_q2, q2[i])
  if expand: return q2[0]
  return q2


def atCopyVect(np.ndarray x):
  """\
  copy vectors (z = x)

  ndarray[3] atCopyVect(ndarray[3])
  ndarray[N,3] atCopyVect(ndarray[N,3])
  """
  if x.ndim == 1 and x.shape[0] == 3:
    return x.copy()
  elif x.ndim == 2 and x.shape[1] == 3:
    return x.copy()
  else:
    raise ValueError("Invalid dimention")


def atInvVect(np.ndarray x):
  """\
  calc inverse vector  (RVSVCT)

  ndarray[3] atInvVect(ndarray[3])
  ndarray[N,3] atInvVect(ndarray[N,3])
  """
  if x.ndim == 1 and x.shape[0] == 3:
    return -x
  elif x.ndim == 2 and x.shape[1] == 3:
    return -x
  else:
    raise ValueError("Invalid dimention: x")


def atAddVect(np.ndarray x, np.ndarray y):
  """\
  add vectors (z = x + y)

  ndarray[3] atAddVect(ndarray[3], ndarray[3])
  ndarray[N,3] atAddVect(ndarray[N,3], ndarray[3])
  ndarray[N,3] atAddVect(ndarray[3], ndarray[N,3])
  ndarray[N,3] atAddVect(ndarray[N,3], ndarray[N,3])
  """
  expand = False
  if x.ndim == 1 and y.ndim == 1:
    expand = True
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if y.ndim == 1:
    y = y.reshape((1, 3))
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if y.ndim != 2:
    raise ValueError("Invalid dimention: y")
  if not valid_array_len(x, y):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(x.shape[0], y.shape[0])
  cdef unsigned int i
  cdef np.ndarray z = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    z[i] = x[min(i, x.shape[0] - 1)] + y[min(i, y.shape[0] - 1)]
  if expand: return z[0]
  return z


def atSubVect(np.ndarray x, np.ndarray y):
  """\
  subtract vectors (z = x - y)

  ndarray[3] atSubVect(ndarray[3], ndarray[3])
  ndarray[N,3] atSubVect(ndarray[N,3], ndarray[3])
  ndarray[N,3] atSubVect(ndarray[3], ndarray[N,3])
  ndarray[N,3] atSubVect(ndarray[N,3], ndarray[N,3])
  """
  expand = False
  if x.ndim == 1 and y.ndim == 1:
    expand = True
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if y.ndim == 1:
    y = y.reshape((1, 3))
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if y.ndim != 2:
    raise ValueError("Invalid dimention: y")
  if not valid_array_len(x, y):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(x.shape[0], y.shape[0])
  cdef unsigned int i
  cdef np.ndarray z = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    z[i] = x[min(i, x.shape[0] - 1)] - y[min(i, y.shape[0] - 1)]
  if expand: return z[0]
  return z


def atMulVect(f, np.ndarray x):
  """\
  multiply scaler for vector (z = f*x)

  ndarray[3] atMulVect(double, ndarray[3])
  ndarray[N,3] atMulVect(ndarray[N], ndarray[3])
  ndarray[N,3] atMulVect(double, ndarray[N,3])
  ndarray[N,3] atMulVect(ndarray[N], ndarray[N,3])
  """
  expand = False
  if is_number(f) and x.ndim == 1:
    expand = True
  f = n2np(f)
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if f.ndim != 1:
    raise ValueError("Invalid dimention: f")
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(f, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(f.shape[0], x.shape[0])
  cdef unsigned int i
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    y[i] = f[min(i, f.shape[0] - 1)] * x[min(i, f.shape[0] - 1)]
  if expand: return y[0]
  return y

def atDivVect(f, np.ndarray x):
  """\
  divide scaler for vector (z = (1/d)*x)

  ndarray[3] atDivVect(double, ndarray[3])
  ndarray[N,3] atDivVect(ndarray[N], ndarray[3])
  ndarray[N,3] atDivVect(double, ndarray[N,3])
  ndarray[N,3] atDivVect(ndarray[N], ndarray[N,3])
  """
  expand = False
  if is_number(f) and x.ndim == 1:
    expand = True
  f = n2np(f)
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if f.ndim != 1:
    raise ValueError("Invalid dimention: f")
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(f, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(f.shape[0], x.shape[0])
  cdef unsigned int i
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    y[i] = x[min(i, f.shape[0] - 1)] / f[min(i, f.shape[0] - 1)]
  if expand: return y[0]
  return y


def atMulAddVect(f, np.ndarray x, g, np.ndarray y):
  """\
  multiply and add vectors (z = f*x + g*y)

  ndarray[3] atMulAddVect(double|ndarray[1], ndarray[3], double|ndarray[1], ndarray[3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[3], double|ndarray[1], ndarray[3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[N,3], double|ndarray[1], ndarray[3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[3], ndarray[N], ndarray[3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[3], double|ndarray[1], ndarray[N,3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[N,3], ndarray[N], ndarray[3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[N,3], double|ndarray[1], ndarray[N,3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[N,3], double|ndarray[1], ndarray[3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[N,3], ndarray[N], ndarray[3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[N,3], double|ndarray[1], ndarray[N,3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[3], ndarray[N], ndarray[3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[3], ndarray[N], ndarray[N,3])
  ndarray[N,3] atMulAddVect(double|ndarray[1], ndarray[3], double|ndarray[1], ndarray[N,3])
  ndarray[N,3] atMulAddVect(ndarray[N], ndarray[N,3], ndarray[N], ndarray[N])
  """
  cdef np.ndarray a = atMulVect(f, x)
  cdef np.ndarray b = atMulVect(g, y)
  return atAddVect(a, b)


def atAngDistance(np.ndarray x, np.ndarray y):
  """\
  calc angular distance (RIKAKU)

  double atAngDistance(np.ndarray[3], np.ndarray[3])
  np.ndarray[N] atAngDistance(np.ndarray[3,N], np.ndarray[3])
  np.ndarray[N] atAngDistance(np.ndarray[3], np.ndarray[N,3])
  np.ndarray[N] atAngDistance(np.ndarray[N,3], np.ndarray[N,3])
  """
  expand = False
  if x.ndim == 1 and y.ndim == 1:
    expand = True
  if x.ndim == 1:
    x = x.reshape(1, 3)
  if y.ndim == 1:
    y = y.reshape(1, 3)
  if x.ndim != 2:
    raise ValueError("Invalid distance: x")
  if y.ndim != 2:
    raise ValueError("Invalid distance: y")
  if not valid_array_len(x, y):
    raise ValueError("Invalid array length")
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  cdef double c_r
  cdef np.ndarray r
  cdef unsigned int n, i
  n = max(x.shape[0], y.shape[0])
  r = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    toVect(y[min(i, y.shape[0] - 1)], at_y)
    at_.atAngDistance(at_x, at_y, &c_r)
    r[i] = c_r
  if expand: return r[0]
  return r


def atCrossPts(np.ndarray x, r1, np.ndarray y, r2):
  """\
  calc crossing points of cones (CROSS)

  AtVect[2] atCrossPts(AtVect, double, AtVect, double)
    :
  AtVect[][2] atCrossPts(AtVect[], double[], AtVect[], double[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_z[2]
  cdef at_.AtVect at_x, at_y
  expand = False
  if x.ndim == 1 and is_number(r1) and y.ndim == 1 and is_number(r2):
    expand = True
  if x.ndim == 1:
    x = x.reshape((1, 3))
  r1 = n2np(r1)
  if y.ndim == 1:
    y = y.reshape((1, 3))
  r2 = n2np(r2)
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if r1.ndim != 1:
    raise ValueError("Invalid dimention: r1")
  if y.ndim != 2:
    raise ValueError("Invalid dimention: y")
  if r2.ndim != 1:
    raise ValueError("Invalid dimention: r2")
  if not valid_array_len(x, r1, y, r2):
    raise ValueError("Invalid array length")
  n = max(x.shape[0], r1.shape[0], y.shape[0], r2.shape[0])
  cdef np.ndarray z = np.ndarray((n, 2, 3), dtype=np.double)
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    toVect(y[min(i, y.shape[0] - 1)], at_y)
    at_.atCrossPts(
      at_x,
      r1[min(i, r1.shape[0] - 1)],
      at_y,
      r2[min(i, r2.shape[0] - 1)],
      at_z
    )
    fromVect(at_z[0], z[i][0])
    fromVect(at_z[1], z[i][1])
  if expand: return z[0]
  return z

def atVectProd(np.ndarray x, np.ndarray y):
  """\
  calc Vector Product (Gaiseki) of two vectors (SPOLE)

  AtVect atVectProd(AtVect, AtVect)
   :
  AtVect[] atVectProd(AtVect[], AtVect[])
  """
  expand = False
  if x.ndim == 1 and y.ndim == 1:
    expand = True
  if x.ndim == 1:
    x = x.reshape(1, 3)
  if y.ndim == 1:
    y = y.reshape(1, 3)
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if y.ndim != 2:
    raise ValueError("Invalid dimention: y")
  if not valid_array_len(x, y):
    raise ValueError("Invalid array length")
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  cdef at_.AtVect at_z
  cdef np.ndarray z
  cdef unsigned int n, i
  n = max(x.shape[0], y.shape[0])
  z = np.ndarray((n, 3), dtype=np.double)
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    toVect(y[min(i, y.shape[0] - 1)], at_y)
    at_.atVectProd(at_x, at_y, at_z)
    fromVect(at_z, z[i])
  if expand: return z[0]
  return z

def atScalProd(np.ndarray x, np.ndarray y):
  """\
  calc Scalar Product (Naiseki, Dot-prooduct) of two vectors

  double atScalProd(AtVect, AtVect)
   :
  double[] atScalProd(AtVect[], AtVect[])
  """
  expand = False
  if x.ndim == 1 and y.ndim == 1:
    expand = True
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if y.ndim == 1:
    y = y.reshape((1, 3))
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if y.ndim != 2:
    raise ValueError("Invlid dimention: y")
  if not valid_array_len(x, y):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(x.shape[0], y.shape[0])
  cdef np.ndarray d = np.zeros((n, ), dtype=np.double)
  cdef unsigned int i
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    toVect(y[min(i, y.shape[0] - 1)], at_y)
    d[i] = at_.atScalProd(at_x, at_y)
  if expand: return d[0]
  return d


def atSetEuler(z, y):
  """\
  set Euler angles (EURST2)

  AtEulerAng atSetEuler(AtPolarVect, AtPolarVect)
   :
  AtEulerAng[] atSetEuler(AtPolarVect[], AtPolarVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtPolarVect at_z
  cdef at_.AtPolarVect at_y
  cdef at_.AtEulerAng at_ea
  expand = False
  if z.ndim == 0 and y.ndim == 0:
    expand = True
  if z.ndim == 0:
    z = z.reshape(1)
  if y.ndim == 0:
    y = y.reshape(1)
  if z.ndim != 1:
    raise ValueError("Invalid dimention: z")
  if y.ndim != 1:
    raise ValueError("Invalid dimention: y")
  if not valid_array_len(z, y):
    raise ValueError("Invalid array length")
  n = max(z.shape[0], y.shape[0])
  cdef np.ndarray ea = np.zeros((n, ), dtype=AtEulerAng)
  for i in range(n):
    toPolarVect(z[min(i, z.shape[0] - 1)], &at_z)
    toPolarVect(y[min(i, y.shape[0] - 1)], &at_y)
    at_.atSetEuler(&at_z, &at_y, &at_ea)
    fromEulerAng(&at_ea, ea[i])
  if expand: return ea[0]
  return ea


def atRotPVect(ea, x):
  """\
  rotate polar vector with Euler angles (ROTAT2)

  AtPolarVect atRotPVect(AtEulerAng, AtPolarVect)
   :
  AtPolarVect[] atRotPVect(AtEulerAng[], AtPolarVect[])
  """
  expand = False
  if ea.ndim == 0 and x.ndim == 0:
    expand = True
  if ea.ndim == 0:
    ea = ea.reshape(1)
  if x.ndim == 0:
    x = x.reshape(1)
  if ea.ndim != 1:
    raise ValueError("Invalid dimention: ea")
  if x.ndim != 1:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(ea, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(ea.shape[0], x.shape[0])
  cdef np.ndarray y = np.zeros((n, ), dtype=AtPolarVect)
  cdef at_.AtEulerAng at_ea
  cdef at_.AtPolarVect at_x
  cdef at_.AtPolarVect at_y
  for i in range(n):
    toEulerAng(ea[min(i, ea.shape[0] - 1)], &at_ea)
    toPolarVect(x[min(i, x.shape[0] - 1)], &at_x)
    at_.atRotPVect(&at_ea, &at_x, &at_y)
    fromPolarVect(&at_y, y[i])
  if expand: return y[0]
  return y


def atVectToPol(np.ndarray x):
  """\
  convert coordinate from Cartesian to polar (RECPOL)

  AtPolarVect atVectToPol(AtVect)
  AtPolarVect[] atVectToPol(AtVect[])
  """
  expand = False
  if x.ndim == 1:
    expand = True
    x = x.reshape(1, 3)
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  cdef unsigned int n = x.shape[0]
  cdef np.ndarray y = np.zeros((n, ), dtype=AtPolarVect)
  cdef unsigned int i
  cdef at_.AtVect at_x
  cdef at_.AtPolarVect at_y
  for i in range(n):
    toVect(x[i], at_x)
    at_.atVectToPol(at_x, &at_y)
    fromPolarVect(&at_y, y[i])
  if expand: return y[0]
  return y

def atPolToVect(x):
  """\
  convert coordinate from polar to Cartesian(POLREC)

  AtVect atPolToVect(AtPolarVect)
   :
  AtVect[] atPolToVect(AtPolarVect[])
  """
  expand = False
  if x.ndim == 0:
    expand = True
    x = x.reshape(1)
  if x.ndim != 1:
    raise ValueError("Invalid dimention: x")
  cdef unsigned int n = x.shape[0]
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  cdef unsigned int i
  cdef at_.AtPolarVect at_x
  cdef at_.AtVect at_y
  for i in range(n):
    toPolarVect(x[i], &at_x)
    at_.atPolToVect(&at_x, at_y)
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y

def atConvPol(x):
  """\
  convert R.A. and Dec from hh.mm.ss to radian

  AtPolarVect atConvPol(AtPolarVect60)
   :
  AtPolarVect[] atConvPol(AtPolarVect60[])
  """
  expand = False
  if x.ndim == 0:
    x = x.reshape(1)
    expand = True
  if x.ndim != 1:
    raise ValueError("Invalid dimention: x")
  cdef unsigned int i, n
  n = x.shape[0]
  cdef np.ndarray y = np.zeros((n, ), dtype=AtPolarVect)
  cdef at_.AtPolarVect60 at_x
  cdef at_.AtPolarVect at_y
  for i in range(n):
    toPolarVect60(x[i], &at_x)
    at_.atConvPol(&at_x, &at_y)
    fromPolarVect(&at_y, y[i])
  if expand: return y[0]
  return y

#
# Time related Functions
#

def atAtTimeToAtTimeD(attime):
  """\
  convert AtTime to AtTimeD

  AtTimeD atAtTimeToAtTimeD(AtTime)
   :
  AtTimeD[] atAtTimeToAtTimeD(AtTime[])
  """
  expand = False
  if attime.ndim == 0:
    attime = attime.reshape(1)
    expand = True
  if attime.ndim != 1:
    raise ValueError("Invalid dimention: attime")
  cdef unsigned int i, n
  n = attime.shape[0]
  cdef np.ndarray attimeD = np.zeros((n, ), dtype=AtTimeD)
  cdef at_.AtTime c_attime
  cdef at_.AtTimeD c_attimeD
  for i in range(n):
    toAtTime(attime[i], &c_attime)
    at_.atAtTimeToAtTimeD(&c_attime, &c_attimeD)
    fromAtTimeD(&c_attimeD, attimeD[i])
  if expand: return attimeD[0]
  return attimeD

def atAtTimeDToAtTime(attimeD):
  """\
  convert AtTimeD to AtTime

  AtTime atAtTimeDToAtTime(AtTimeD)
  AtTime[] atAtTimeDToAtTime(AtTimeD[])
  """
  expand = False
  if attimeD.ndim == 0:
    attimeD = attimeD.reshape(1)
    expand = True
  if attimeD.ndim != 1:
    raise ValueError("Invalid dimention: attimeD")
  cdef unsigned int i, n
  n = attimeD.shape[0]
  cdef np.ndarray attime = np.zeros((n, ), dtype=AtTime)
  cdef at_.AtTime c_attime
  cdef at_.AtTimeD c_attimeD
  for i in range(n):
    toAtTimeD(attimeD[i], &c_attimeD)
    at_.atAtTimeDToAtTime(&c_attimeD, &c_attime)
    fromAtTime(&c_attime, attime[i])
  if expand: return attime[0]
  return attime

def atReformatAtTime(attime):
  """\
  reformat AtTime, as 0<=ss<1, 0<=sc<=59, 0<=mn<=59, 0<=hr<=23, etc

  AtTime atReformatAtTime(AtTime)
  AtTime[] atReformatAtTime(AtTime[])
  AtTimeD atReformatAtTime(AtTimeD)
  AtTimeD[] atReformatAtTime(AtTimeD[])
  """
  expand = False
  if attime.ndim == 0:
    attime = attime.reshape(1)
    expand = True
  if attime.ndim != 1:
    raise ValueError("Inavlid dimention: attime")
  cdef unsigned int i, n
  n = attime.shape[0]
  cdef np.ndarray attime2
  cdef at_.AtTime c_attime
  cdef at_.AtTimeD c_attimeD
  dtype = AtTime
  try:
    _ = attime['ms']
  except ValueError:
    dtype = AtTimeD
  if dtype == AtTime:
    attime2 = np.zeros((n, ), dtype=AtTime)
    for i in range(n):
      toAtTime(attime[i], &c_attime)
      at_.atReformatAtTime(&c_attime)
      fromAtTime(&c_attime, attime2[i])
  else:
    attime2 = np.zeros((n, ), dtype=AtTimeD)
    for i in range(n):
      toAtTimeD(attime[i], &c_attimeD)
      at_.atReformatAtTimeD(&c_attimeD)
      fromAtTimeD(&c_attimeD, attime2[i])
  if expand: return attime2[0]
  return attime2

def atReformatAtTimeD(attimeD):
  """\
  reformat AtTime, as 0<=ss<1, 0<=sc<=59, 0<=mn<=59, 0<=hr<=23, etc

  AtTimeD atReformatAtTimeD(AtTimeD)
  AtTimeD[] atReformatAtTimeD(AtTimeD[])
  """
  return atReformatAtTime(attimeD)

def atCTime(time):
  """\
  format time in character (yy/mm/dd hh:mm:ss.ssssss) char[25]

  str atCTime(AtTime)
  str[] atCTime(AtTime[])
  str atCTime(AtTimeD)
  str[] atCTime(AtTimeD[])
  """
  expand = False
  if time.ndim == 0:
    time = time.reshape(1)
    expand = True
  if time.ndim != 1:
    raise ValueError("Inavlid dimention: time")
  cdef unsigned int i, n
  n = time.shape[0]
  cdef np.ndarray ctime = np.zeros((n, ), dtype='U25')
  cdef at_.AtTime c_time
  cdef at_.AtTimeD c_timeD
  cdef char c_ctime[25]
  dtype = AtTime
  try:
    _ = time['ms']
  except ValueError:
    dtype = AtTimeD
  if dtype == AtTime:
    for i in range(n):
      toAtTime(time[i], &c_time)
      at_.atCTime(&c_time, c_ctime)
      ctime[i] = c_ctime
  else:
    for i in range(n):
      toAtTimeD(time[i], &c_timeD)
      at_.atCTimeD(&c_timeD, c_ctime)
      ctime[i] = c_ctime
  if expand: return ctime[0]
  return ctime

def atCTimeD(time):
  """\
  format time in character (yy/mm/dd hh:mm:ss.ssssss) char[25]

  str atCTime(AtTimeD)
  str[] atCTime(AtTimeD[])
  """
  return atCTime(time)

def atCTime2(time):
  """\
  format time in character (yyyy/mm/dd hh:mm:ss.ssssss) char[27]

  str atCTime2(AtTime)
  str[] atCTime2(AtTime[])
  str atCTime2(AtTimeD)
  str[] atCTime2(AtTimeD[])
  """
  expand = False
  if time.ndim == 0:
    time = time.reshape(1)
    expand = True
  if time.ndim != 1:
    raise ValueError("Inavlid dimention: time")
  cdef unsigned int i, n
  n = time.shape[0]
  cdef np.ndarray ctime = np.zeros((n, ), dtype='U27')
  cdef at_.AtTime c_time
  cdef at_.AtTimeD c_timeD
  cdef char c_ctime[25]
  dtype = AtTime
  try:
    _ = time['ms']
  except ValueError:
    dtype = AtTimeD
  if dtype == AtTime:
    for i in range(n):
      toAtTime(time[i], &c_time)
      at_.atCTime2(&c_time, c_ctime)
      ctime[i] = c_ctime
  else:
    for i in range(n):
      toAtTimeD(time[i], &c_timeD)
      at_.atCTimeD2(&c_timeD, c_ctime)
      ctime[i] = c_ctime
  if expand: return ctime[0]
  return ctime

def atCTimeD2(time):
  """\
  format time in character (yyyy/mm/dd hh:mm:ss.ssssss) char[27]

  str atCTimeD2(AtTimeD)
  str[] atCTimeD2(AtTimeD[])
  """
  return atCTime2(time)

def atMJulian(time):
  """\
  convert UT to Modified Julian Day (MJULIA)

  double atMJulian(AtTime)
  double[] atMJulian(AtTime[])
  """
  expand = False
  if time.ndim == 0:
    time = time.reshape(1)
    expand = True
  if time.ndim != 1:
    raise ValueError("Inavlid dimention: time")
  cdef unsigned int i, n
  n = time.shape[0]
  cdef np.ndarray mjd = np.zeros((n, ), dtype='f8')
  cdef at_.AtTime c_time
  cdef at_.AtTimeD c_timeD
  cdef double c_mjd
  dtype = AtTime
  try:
    _ = time['ms'] # AtTime
  except ValueError:
    dtype = AtTimeD
  if dtype == AtTime:
    for i in range(n):
      toAtTime(time[i], &c_time)
      at_.atMJulian(&c_time, &c_mjd)
      mjd[i] = c_mjd
  else:
    for i in range(n):
      toAtTimeD(time[i], &c_timeD)
      at_.atMJulianD(&c_timeD, &c_mjd)
      mjd[i] = c_mjd
  if expand: return mjd[0]
  return mjd

def atMJulianD(np.ndarray time):
  """\
  convert UT to Modified Julian Day (MJULIA)

  double atMJulian(AtTimeD)
  double[] atMJulian(AtTimeD[])
  """
  return atMJulian(time)

def atMJDate(mjd):
  """\
  convert Modified Julian Day to UT (MJDATE)

  AtTime atMJDate(double)
  AtTime[] atMJDate(double[])
  """
  expand = False
  if is_number(mjd):
    mjd = n2np(mjd)
    expand = True
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int i, n
  n = mjd.shape[0]
  cdef np.ndarray time = np.zeros((n, ), dtype=AtTime)
  cdef at_.AtTime c_time
  for i in range(n):
    at_.atMJDate(mjd[i], &c_time)
    fromAtTime(&c_time, time[i])
  if expand: return time[0]
  return time

def atMJDateD(mjd):
  """\
  convert Modified Julian Day to UT (MJDATE)

  AtTimeD atMJDate(double)
  AtTimeD[] atMJDate(double[])
  """
  expand = False
  if is_number(mjd):
    mjd = n2np(mjd)
    expand = True
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int i, n
  n = mjd.shape[0]
  cdef np.ndarray time = np.zeros((n, ), dtype=AtTimeD)
  cdef at_.AtTimeD c_time
  for i in range(n):
    at_.atMJDateD(mjd[i], &c_time)
    fromAtTimeD(&c_time, time[i])
  if expand: return time[0]
  return time

def atMissionTimeResetTable():
    """\
    reset internal table
    """
    at_.atMissionTimeResetTable()

def atMissionTimeInit(str filename, int flag=1):
    """\
    read leapsec table (or just return file name)
    """
    b_filename = filename.encode('UTF-8')
    cdef const char *c_fname = at_.atMissionTimeInit(b_filename, flag)
    if c_fname is None:
      raise Exception("atMissionTimeInit failed")
    fname = c_fname.decode('UTF-8')
    return fname

def atAtTimeDToMission(mjd_base, attime):
  """\
  convert AtTimeD into mission time

  double atAtTimeDToMission(double, AtTime)
   :
  double[] atAtTimeDToMission(double[], AtTime[])
  """
  expand = False
  if is_number(mjd_base) and attime.ndim == 0:
    expand = True
  mjd_base = n2np(mjd_base)
  if mjd_base.ndim != 1:
    raise ValueError("Invalid dimention: mjd_base")
  if attime.ndim == 0:
    attime = attime.reshape(1)
  if attime.ndim != 1:
    raise ValueError("Invalid dimention: attime")
  if not valid_array_len(mjd_base, attime):
    raise ValueError("Invalid array length")
  cdef unsigned int i, n
  n = max(mjd_base.shape[0], attime.shape[0])
  cdef np.ndarray mission = np.zeros((n, ), dtype=np.double)
  cdef at_.AtTimeD c_attime
  cdef double c_mission
  for i in range(n):
    toAtTimeD(attime[min(i, attime.shape[0]- 1)], &c_attime)
    at_.atAtTimeDToMission(mjd_base[min(i, mjd_base.shape[0] - 1)], &c_attime, &c_mission)
    mission[i] = c_mission
  if expand: return mission
  return mission

def atMissionToAtTimeD(mjd_base, mission):
  """\
  convert mission time to AtTimeD

  AtTimeD atMissionToAtTimeD(double, double)
   :
  AtTimeD[] atMissionToAtTimeD(double[], double[])
  """
  expand = False
  if is_number(mjd_base) and is_number(mission):
    expand = True
  mjd_base = n2np(mjd_base)
  mission = n2np(mjd_base)
  if mjd_base.ndim != 1:
    raise ValueError("Invalid dimention: mjd_base")
  if mission.ndim != 1:
    raise ValueError("Invalid dimention: mission")
  if not valid_array_len(mjd_base, mission):
    raise ValueError("Invalid array length")
  cdef unsigned int i, n
  n = max(mjd_base.shape[0], mission.shape[0])
  cdef np.ndarray attime = np.zeros((n, ), dtype=AtTimeD)
  cdef at_.AtTimeD c_attime
  for i in range(n):
    at_.atMissionToAtTimeD(
      mjd_base[min(i, mjd_base.shape[0] - 1)],
      mission[min(i, mission.shape[0] - 1)],
      &c_attime)
    fromAtTimeD(&c_attime, attime[i])
  if expand: return attime[0]
  return attime

def atMJDToMission(mjd_base, mjd):
  """\
  convert Modified Julian Day to mission time
  double atMJDToMission(double, double)
  ndarray[N] atMJDToMission(ndarray[N], double)
  ndarray[N] atMJDToMission(double, ndarray[N])
  ndarray[N] atMJDToMission(double[N], ndarray[N])
  """
  expand = False
  if is_number(mjd_base) and is_number(mjd):
    expand = True
  mjd_base = n2np(mjd_base)
  mjd = n2np(mjd)
  if mjd_base.ndim != 1:
    raise ValueError("Invalid dimention: mjd_base")
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if not valid_array_len(mjd_base, mjd):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd_base.shape[0], mjd.shape[0])
  cdef np.ndarray mission_time = np.zeros((n, ), dtype=np.double)
  cdef double c_mission_time
  cdef unsigned int i
  for i in range(n):
    at_.atMJDToMission(
      mjd_base[min(i, mjd_base.shape[0] - 1)],
      mjd[min(i, mjd.shape[0] - 1)],
      &c_mission_time)
    mission_time[i] = c_mission_time
  if expand: return mission_time[0]
  return mission_time

def atMissionToMJD(mjd_base, mission_time):
  """\
  convert mission time to Modified Julain Day

  double atMissionToMJD(double, double)
  ndarray[N] atMissionToMJD(ndaray[N], double)
  ndarray[N] atMissionToMJD(double, ndarray[N])
  ndarray[N] atMissionToMJD(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(mjd_base) and is_number(mission_time):
    expand = True
  mjd_base = n2np(mjd_base)
  mission_time = n2np(mission_time)
  if mjd_base.ndim != 1:
    raise ValueError("Invalid dimention: mjd_base")
  if mission_time.ndim != 1:
    raise ValueError("Invalid dimention: mission_time")
  if not valid_array_len(mjd_base, mission_time):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd_base.shape[0], mission_time.shape[0])
  cdef unsigned int i
  cdef np.ndarray mjd = np.zeros((n, ), dtype=np.double)
  cdef double c_mjd
  for i in range(n):
    at_.atMissionToMJD(
      mjd_base[min(i, mjd_base.shape[0] - 1)],
      mission_time[min(i, mission_time.shape[0] - 1)],
      &c_mjd)
    mjd[i] = c_mjd
  if expand: return mjd[0]
  return mjd



#
# Functions related to ephemeris and orbit
#

def atPrecession(mjd0, np.ndarray x0, mjd):
  """\
  convert equatorial coordinate systems correcting for precession (SAISA)

  ndaray[3] atPrecession(double, ndarray[3], double)
  ndaray[N,3] atPrecession(ndarray[N], ndarray[3], double)
  ndaray[N,3] atPrecession(double, ndarray[N,3], double)
  ndaray[N,3] atPrecession(double, ndarray[3], ndarray[N])
  ndaray[N,3] atPrecession(ndarray[N], ndarray[N,3], double)
  ndaray[N,3] atPrecession(ndarray[N], ndarray[3], ndarray[N])
  ndaray[N,3] atPrecession(ndarray[N], ndarray[N,3], ndarray[N])
  """
  expand = False
  if is_number(mjd0) and x0.ndim == 1 and is_number(mjd):
    expand = True
  mjd0 = n2np(mjd0)
  if x0.ndim == 1:
    x0 = x0.reshape((1, 3))
  mjd = n2np(mjd)
  if mjd0.ndim != 1:
    raise ValueError("Invalid dimention: mjd0")
  if x0.ndim != 2:
    raise ValueError("Invalid dimention: x0")
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if not valid_array_len(mjd0, x0, mjd):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd0.shape[0], x0.shape[0], mjd.shape[0])
  cdef unsigned int i
  cdef np.ndarray x = np.zeros((n, 3), dtype=np.double)
  cdef at_.AtVect at_x0
  cdef at_.AtVect at_x
  for i in range(n):
    toVect(x0[min(i, x0.shape[0] - 1)], at_x0)
    at_.atPrecession(
      mjd0[min(i, mjd0.shape[0] - 1)],
      at_x0,
      mjd[min(i, mjd.shape[0] - 1)],
      at_x)
    fromVect(at_x, x[i])
  if expand: return x[0]
  return x

def atPrecessRM(mjd0, mjd):
  """\
  Find Rotation Matrix for conversion of equatorial coordinate systems
  correcting for precession (SAISA)

  ndarray[3,3] atPrecessRM(double, double)
  ndarray[N,3,3] atPrecessRM(ndarray[N], double)
  ndarray[N,3,3] atPrecessRM(double, ndarray[N])
  ndarray[N,3,3] atPrecessRM(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(mjd0) and is_number(mjd):
    expand = True
  mjd0 = n2np(mjd0)
  mjd = n2np(mjd)
  if mjd0.ndim != 1:
    raise ValueError("Invalid dimention: mjd0")
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if not valid_array_len(mjd0, mjd):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd0.shape[0], mjd.shape[0])
  cdef unsigned int i
  cdef at_.AtRotMat at_rm
  cdef np.ndarray rm = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    at_.atPrecessRM(
      mjd0[min(i, mjd0.shape[0] - 1)],
      mjd[min(i, mjd.shape[0] - 1)],
      at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm

def atPrecessEuler(mjd0, mjd):
  """\
  Find Euler Angles for conversion of equatorial coordinate systems
  correcting for precession (SAISA)
  """
  expand = False
  if is_number(mjd0) and is_number(mjd):
    expand = True
  mjd0 = n2np(mjd0)
  mjd = n2np(mjd)
  if mjd0.ndim != 1:
    raise ValueError("Invalid dimention: mjd0")
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if not valid_array_len(mjd0, mjd):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd0.shape[0], mjd.shape[0])
  cdef unsigned int i
  cdef at_.AtEulerAng at_ea
  cdef np.ndarray ea = np.zeros((n, ), dtype=AtEulerAng)
  for i in range(n):
    at_.atPrecessEuler(
      mjd0[min(i, mjd0.shape[0] - 1)],
      mjd[min(i, mjd.shape[0] - 1)],
      &at_ea
    )
    fromEulerAng(&at_ea, ea[i])
  if expand: return ea[0]
  return ea

def atPrecessRMJ2000(mjd):
  """\
  Equivalent to atPrecessRM(mjd, MJD_J2000, rm)

  ndarray[3,3] atPrecessRMJ2000(double)
  ndarray[N,3,3] atPrecessRMJ2000(ndarray[N])
  """
  return atPrecessEuler(mjd, MJD_J2000)

def atSaisa(mjd0, pv0, mjd):
  """\
  conversion of equatorial coordinate systems
  correcting for precession (based on SAISA in ASTRO-C csub1)

  AtPolarVect atSaisa(double, AtPolarVect, double)
  AtPolarVect[] atSaisa(double[], AtPolarVect[], double[])
  """
  expand = False
  if is_number(mjd0) and pv0.ndim == 0 and is_number(mjd):
    expand = True
  mjd0 = n2np(mjd0)
  if pv0.ndim == 0:
    pv0 = pv0.reshape(1)
  mjd = n2np(mjd)
  if mjd0.ndim != 1:
    raise ValueError("Invalid dimention: mjd0")
  if pv0.ndim != 1:
    raise ValueError("Invalid dimention: pv0")
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if not valid_array_len(mjd0, pv0, mjd):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd0.shape[0], pv0.shape[0], mjd.shape[0])
  cdef unsigned int i
  cdef at_.AtPolarVect at_pv0
  cdef at_.AtPolarVect at_pv
  cdef np.ndarray pv = np.zeros((n, ), dtype=AtPolarVect)
  for i in range(n):
    toPolarVect(pv0[min(i, pv0.shape[0] - 1)], &at_pv0)
    at_.atSaisa(
      mjd0[min(i, mjd0.shape[0] - 1)],
      &at_pv0,
      mjd[min(i, mjd.shape[0] - 1)],
      &at_pv
    )
    fromPolarVect(&at_pv, pv[i])
  if expand: return pv[0]
  return pv

def atSetElement3(str filename, double mjd0, int kchk):
  """\
  set orbital elements, with mjdz <= mjd0 < mjdn, see atElementTime3()

  AtElement3 atSetElement3(str, double, int)
  """
  b_filename = filename.encode('UTF-8')
  cdef at_.AtElement3 c_el
  cdef np.ndarray el = np.zeros((), dtype=AtElement3)
  at_.atSetElement3(&c_el, b_filename, mjd0, kchk)
  fromAtElement3(&c_el, el)
  return el

def atElementTime3(np.ndarray el):
  """\
  get time of orbital elements, with mjdz <= mjd0 < mjdn, see atSetElement3()
  """
  return el['mjdz'], el['mjdn']

def atPathNum3(np.ndarray el, mjd):
  """\
  calc path number (PATNUM)

  str atPathNum3(AtElement3, duble)
  str[] atPathNum3(AtElement3, duble[])
  """
  expand = False
  if is_number(mjd):
    mjd = n2np(mjd)
    expand = True
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int i, n
  n = mjd.shape[0]
  cdef at_.AtElement3 c_el
  toAtElement3(el, &c_el)
  cdef np.ndarray path = np.zeros((n, ), dtype='U11')
  cdef char c_path[11]
  for i in range(n):
    at_.atPathNum3(&c_el, mjd[i], c_path)
    path[i] = c_path
  if expand: return path[0]
  return path

def atPathNumUSC3(np.ndarray el, mjd):
  """\
  calc JAXA style path number "YYMMDDnnnn" at USC 34m station

  str atPathNum3(AtElement3, duble)
  str[] atPathNum3(AtElement3, duble[])
  """
  expand = False
  if is_number(mjd):
    mjd = n2np(mjd)
    expand = True
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int i, n
  n = mjd.shape[0]
  cdef at_.AtElement3 c_el
  toAtElement3(el, &c_el)
  cdef np.ndarray path = np.zeros((n, ), dtype='U11')
  cdef char c_path[11]
  for i in range(n):
    at_.atPathNumUSC3(&c_el, mjd[i], c_path)
    path[i] = c_path
  if expand: return path[0]
  return path

def atSatPos3(np.ndarray el, mjd):
  """\
  calc satellite position (SATPNT) with the Earth center as origin

  AtVect atSatPos3(AtElement3, double)
  AtVect[] atSatPos3(AtElement3, double[])
  """
  expand = False
  if is_number(mjd):
    mjd = n2np(mjd)
    expand = True
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int i, n
  n = mjd.shape[0]
  cdef at_.AtElement3 c_el
  toAtElement3(el, &c_el)
  cdef np.ndarray x = np.zeros((n, 3), dtype='f8')
  cdef at_.AtVect c_x
  for i in range(n):
    at_.atSatPos3(&c_el, mjd[i], c_x)
    fromVect(c_x, x[i])
  if expand: return x[0]
  return x


def atGeodcr(mjd, np.ndarray x):
  """\
  convert sidereal position to geodetic/geographical position
  on earth (GEODCR)

  double, double, double atGeodcr(double, AtVect)
   :
  double[], double[], double[] atGeodcr(double[], AtVect[])
  """
  expand = False
  if is_number(mjd) and x.ndim == 1:
    expand = True
  mjd = n2np(mjd)
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(mjd, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd.shape[0], x.shape[0])
  cdef unsigned int i
  cdef np.ndarray heigh = np.zeros((n, ), dtype='f8')
  cdef np.ndarray longi = np.zeros((n, ), dtype='f8')
  cdef np.ndarray latt = np.zeros((n, ), dtype='f8')
  cdef double c_heigh, c_longi, c_latt
  cdef at_.AtVect at_x
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    at_.atGeodcr(
      mjd[min(i, mjd.shape[0] - 1)],
      at_x,
      &c_heigh,
      &c_longi,
      &c_latt
    )
    heigh[i] = c_heigh
    longi[i] = c_longi
    latt[i] = c_latt
  if expand: return heigh[0], longi[0], latt[0]
  return heigh, longi, latt

def atGeodetic(mjd, np.ndarray x):
  """\
  AtVect atGeodetic(double, AtVect)
   :
  AtVect[] atGeodetic(double[], AtVect[])
  """
  expand = False
  if is_number(mjd) and x.ndim == 1:
    expand = True
  mjd = n2np(mjd)
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(mjd, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd.shape[0], x.shape[0])
  cdef unsigned int i
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    at_.atGeodetic(mjd[min(i, mjd.shape[0] - 1)], at_x, at_y)
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y

def atInvGeodetic(mjd, np.ndarray x):
  """\
  ndarray[3] atInvGeodetic(double, ndarray[3])
  ndarray[N,3] atInvGeodetic(ndarray[N], ndarray[3])
  ndarray[N,3] atInvGeodetic(double, ndarray[N,3])
  ndarray[N,3] atInvGeodetic(ndarray[N], ndarray[N,3])
  """
  expand = False
  if is_number(mjd) and x.ndim == 1:
    expand = True
  mjd = n2np(mjd)
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if not valid_array_len(mjd, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(mjd.shape[0], x.shape[0])
  cdef unsigned int i
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    at_.atInvGeodetic(mjd[min(i, mjd.shape[0] - 1)], at_x, at_y)
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y

def atEllipsoid(xp):
  """\
  convert polar geodetic coordinate latitude radial distance
  to the geographic latitude and altitude from the earth surface
  correcting for the ellipsoidal shape of the earth

  double, double atEllipsoid(AtEulerAng)
  double[], double[] atEllipsoid(AtEulerAng[])
  """
  expand = False
  if xp.ndim == 0:
    expand = True
    xp = xp.reshape(1)
  if xp.ndim != 1:
    raise ValueError("Invalid dimention: xp")
  cdef unsigned int n = xp.shape[0]
  cdef unsigned int i
  cdef double c_latt, c_heigh
  cdef at_.AtPolarVect at_xp
  cdef np.ndarray pos = np.zeros((n, ), dtype=[('latt', 'f8'), ('heigh', 'f8')])
  cdef np.ndarray latt = np.zeros((n, ), dtype='f8')
  cdef np.ndarray heigh = np.zeros((n, ), dtype='f8')
  for i in range(n):
    toPolarVect(xp[i], &at_xp)
    at_.atEllipsoid(&at_xp, &c_latt, &c_heigh)
    latt[i] = c_latt
    heigh[i] = c_heigh
  if expand: return latt[0], heigh[0]
  return latt, heigh


def atSetGeoRM(mjd):
  """\
  calc rotation matrix for converting equatorial position (J2000)
  to the geocentric position on earth

  AtRotMat atSetGeoRM(double)
  AtRotMat[] atSetGeoRM(double[])
  """
  expand = False
  if is_number(mjd):
    expand = True
    mjd = n2np(mjd)
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int n = mjd.shape[0]
  cdef unsigned int i
  cdef at_.AtRotMat at_rm
  cdef np.ndarray rm = np.zeros((n, 3, 3), dtype=np.double)
  rm = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    at_.atSetGeoRM(mjd[i], at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm

def atSidereal(mjd):
  """\
  calc sidereal time (KOSEJI)

  double atSidereal(double)
  double[] atSidereal(double[])
  """
  expand = False
  if is_number(mjd):
    expand = True
    mjd = n2np(mjd)
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  cdef unsigned int n = mjd.shape[0]
  cdef unsigned int i
  cdef np.ndarray gsttod = np.zeros((n, ), dtype=np.double)
  cdef double c_gsttod
  n = mjd.shape[0]
  for i in range(n):
    at_.atSidereal(mjd[i], &c_gsttod)
    gsttod[i] = c_gsttod
  if expand: return gsttod[0]
  return gsttod

def atKepler(g, eccent):
  """\
  solve Kepler equation (KEPLER)  g + e sin E = E

  double atKepler(double, double)
   :
  double[] atKepler(double[], double[])
  """
  expand = False
  if is_number(g) and is_number(eccent):
    expand = True
  g = n2np(g)
  eccent = n2np(eccent)
  if g.ndim != 1:
    raise ValueError("Invalid dimention: g")
  if eccent.ndim != 1:
    raise ValueError("Invalid dimention: eccent")
  cdef unsigned int n = max(g.shape[0], eccent.shape[0])
  cdef unsigned int i
  cdef double c_e
  cdef np.ndarray e = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    at_.atKepler(g[i], eccent[i], &c_e)
    e[i] = c_e
  if expand: return e[0]
  return e

def atOrbPlane(np.ndarray x, ol, ob, ai):
  """\
  coordinate conversion from orbital plane to celestial (SATINA)

  AtVect atOrbPlane(AtVect, double, double, double)
   :
  AtVect[] atOrbPlane(AtVect[], double[], double[], double[])
  """
  expand = False
  if x.ndim == 1 and is_number(ol) and is_number(ob) and is_number(ai):
    expand = True
  if x.ndim == 1:
    x = x.reshape((1 ,3))
  ol = n2np(ol)
  ob = n2np(ob)
  ai = n2np(ai)
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if ol.ndim != 1:
    raise ValueError("Invalid dimention: ol")
  if ob.ndim != 1:
    raise ValueError("Invalid dimention: ob")
  if ai.ndim != 1:
    raise ValueError("Invalid dimention: ai")
  if not valid_array_len(x, ol, ob, ai):
    raise ValueError("Invalid array length")
  cdef unsigned int n = max(x.shape[0], ol.shape[0], ob.shape[0], ai.shape[0])
  cdef unsigned int i
  cdef at_.AtVect at_x
  cdef at_.AtVect at_y
  cdef np.ndarray y = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    at_.atOrbPlane(
      at_x,
      ol[min(i, ol.shape[0] - 1)],
      ob[min(i, ob.shape[0] - 1)],
      ai[min(i, ai.shape[0] - 1)],
      at_y
    )
    fromVect(at_y, y[i])
  if expand: return y[0]
  return y

def atGeographic(np.ndarray y):
  """\
  convert geographic location of Tracking Station on Earth to Geodetic

  AtVect atGeographic(AtPolarVect)
  AtVect[] atGeographic(AtPolarVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtPolarVect at_y
  cdef at_.AtVect at_z
  cdef np.ndarray z
  expand = False
  if y.ndim == 0:
    expand = True
    y = y.reshape(1)
  if y.ndim != 1:
    raise ValueError("Invalid dimention y")
  n = y.shape[0]
  z = np.zeros((n ,3), dtype=np.double)
  for i in range(n):
    toPolarVect(y[i], &at_y)
    at_.atGeographic(&at_y, at_z)
    fromVect(at_z, z[i])
  if expand: return z[0]
  return z

def atGeographicToGeodetic(y):
  """\
  same with atGeographic(), calling it internally

  AtVect atGeographicToGeodetic(AtPolarVect)
  AtVect[] atGeographicToGeodetic(AtPolarVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtPolarVect at_y
  cdef at_.AtVect at_z
  cdef np.ndarray z
  expand = False
  if y.ndim == 0:
    expand = True
    y = y.reshape(1)
  if y.ndim != 1:
    raise ValueError("Invalid dimention: y")
  n = y.shape[0]
  z = np.zeros((n ,3), dtype=np.double)
  for i in range(n):
    toPolarVect(y[i], &at_y)
    at_.atGeographicToGeodetic(&at_y, at_z)
    fromVect(at_z, z[i])
  if expand: return z[0]
  return z

def atGeodesic(y):
  """\
  same with atGeographic(), calling it internally

  AtVect atGeodesic(AtPolarVect)
  AtVect[] atGeodesic(AtPolarVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtPolarVect at_y
  cdef at_.AtVect at_z
  cdef np.ndarray z
  expand = False
  if y.ndim == 0:
    expand = True
    y = y.reshape(1)
  if y.ndim != 1:
    raise ValueError("Invalid dimention: y")
  n = y.shape[0]
  z = np.zeros((n ,3), dtype=np.double)
  for i in range(n):
    toPolarVect(y[i], &at_y)
    at_.atGeodesic(&at_y, at_z)
    fromVect(at_z, z[i])
  if expand: return z[0]
  return z

def atGeodeticToGeographic(np.ndarray vect):
  """\
  convert geodetic vector into geographic position on earth surface

  AtPolarVect atGeodeticToGeographic(AtVect)
  AtPolarVect[] atGeodeticToGeographic(AtVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_vect
  cdef at_.AtPolarVect at_pv
  cdef np.ndarray pv
  expand = False
  if vect.ndim == 1:
    expand = True
    vect = vect.reshape((1, 3))
  if vect.ndim != 2:
    raise ValueError("Invalid dimention: vect")
  n = vect.shape[0]
  pv = np.zeros((n, ), dtype=AtPolarVect)
  for i in range(n):
    toVect(vect[i], at_vect)
    at_.atGeodeticToGeographic(at_vect, &at_pv)
    fromPolarVect(&at_pv, pv[i])
  if expand: return pv[0]
  return pv

def atGroundCoord(np.ndarray station):
  """\
  Convert from Geodetic to Ground Coordinate at the Tracking Station

  AtRotMat atGroundCoord(AtVect)
  AtRotMat[] atGroundCoord(AtVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_station
  cdef at_.AtRotMat at_rm
  cdef np.ndarray rm
  expand = False
  if station.ndim == 1:
    station = station.reshape((1, 3))
  if station.ndim != 2:
    raise ValueError("Invalid dimention: station")
  n = station.shape[0]
  rm = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    toVect(station[i], at_station)
    at_.atGroundCoord(at_station, at_rm)
    fromRotMat(at_rm, rm[i])
  if expand: return rm[0]
  return rm

def atAzElSet(np.ndarray y):
  """\
  Set Up geodetic vector to the tracking station and Rotation Matirix
  to convert sidefloat coord to Ground (Earth-bound) coord
  Based on "ADAZEL"

  AtVect, AtRotMat atAzElSet(AtPolarVect)
  AtVect[], AtRotMat[] atAzElSet(AtPolarVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtPolarVect at_y
  cdef at_.AtVect at_vStation
  cdef at_.AtRotMat at_stationRM
  cdef np.ndarray vStation
  cdef np.ndarray stationRM
  expand = False
  if y.ndim == 0:
    expand = True
    y = y.reshape(1)
  if y.ndim != 1:
    raise ValueError("Invalid dimention: y")
  n = y.shape[0]
  vStation = np.zeros((n, 3), dtype=np.double)
  stationRM = np.zeros((n, 3, 3), dtype=np.double)
  for i in range(n):
    toPolarVect(y[i], &at_y)
    at_.atAzElSet(&at_y, at_vStation, at_stationRM)
    fromVect(at_vStation, vStation[i])
    fromRotMat(at_stationRM, stationRM[i])
  if expand: return vStation[0], stationRM[0]
  return vStation, stationRM

def atAzEl(np.ndarray x, np.ndarray stationV, np.ndarray stationRM):
  """\
  Azimuth and Elevation for a vector in Ground (Earth-bound) coord

  AtPolarVect atAzEl(AtVect, AtVect, AtRotMat)
   :
  AtPolarVect[] atAzEl(AtVect[], AtVect[], AtRotMat[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x
  cdef at_.AtVect at_stationV
  cdef at_.AtRotMat at_stationRM
  cdef at_.AtPolarVect at_y
  cdef np.ndarray y
  expand = False
  if x.ndim == 1 and stationV.ndim == 1 and stationRM.ndim == 2:
    expand = True
  if x.ndim == 1:
    x = x.reshape((1, 3))
  if stationV.ndim == 1:
    stationV = stationV.reshape((1, 3))
  if stationRM.ndim == 2:
    stationRM = stationRM.reshape((1, 3, 3))
  if x.ndim != 2:
    raise ValueError("Invalid dimention: x")
  if stationV.ndim != 2:
    raise ValueError("Invalid dimention: stationV")
  if stationRM.ndim != 3:
    raise ValueError("Invalid dimention: stationRM")
  if not valid_array_len(x, stationV, stationRM):
    raise ValueError("Invalid array length")
  n = max(x.shape[0], stationV.shape[0], stationRM.shape[0])
  y = np.zeros((n, ), dtype=AtPolarVect)
  for i in range(n):
    toVect(x[min(i, x.shape[0] - 1)], at_x)
    toVect(stationV[min(i, stationV.shape[0] - 1)], at_stationV)
    toRotMat(stationRM[min(i, stationRM.shape[0] - 1)], at_stationRM)
    at_.atAzEl(at_x, at_stationV, at_stationRM, &at_y)
    fromPolarVect(&at_y, y[i])
  if expand: return y[0]
  return y

def atEarthOccult(np.ndarray satVect, np.ndarray xVect, np.ndarray sunVect):
  """\
  examine earth occultation of specified direction (YEARTH)
  return flag, el

  int, double atEarthOccult(AtVect, AtVect, AtVect)
   :
  int[], double[] atEarthOccult(AtVect[], AtVect[], AtVect[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_satVect
  cdef at_.AtVect at_xVect
  cdef at_.AtVect at_sunVect
  expand = False
  if satVect.ndim == 1 and xVect.ndim == 1 and sunVect.ndim == 1:
    expand = True
  if satVect.ndim == 1:
    satVect = satVect.reshape((1, 3))
  if xVect.ndim == 1:
    xVect = xVect.reshape((1, 3))
  if sunVect.ndim == 1:
    sunVect = sunVect.reshape((1, 3))
  if satVect.ndim != 2:
    raise ValueError("Invalid dimention: satVect")
  if xVect.ndim != 2:
    raise ValueError("Invalid dimention: xVect")
  if sunVect.ndim != 2:
    raise ValueError("Invalid dimention: sunVect")
  if not valid_array_len(satVect, xVect, sunVect):
    raise ValueError("Invalid array length")
  n = max(satVect.shape[0], xVect.shape[0], sunVect.shape[0])
  cdef np.ndarray flag = np.zeros((n, ), dtype='i4')
  cdef np.ndarray el = np.zeros((n, ), dtype='f8')
  cdef int c_flag
  cdef double c_el
  for i in range(n):
    toVect(satVect[min(i, satVect.shape[0] - 1)], at_satVect)
    toVect(xVect[min(i, xVect.shape[0] - 1)], at_xVect)
    toVect(sunVect[min(i, sunVect.shape[0] - 1)], at_sunVect)
    at_.atEarthOccult(at_satVect, at_xVect, at_sunVect, &c_flag, &c_el)
    flag[i] = c_flag
    el[i] = c_el
  if expand:
    return flag[0], el[0]
  return flag, el

cdef class AtRigData2:
  cdef at_.AtRigData2 *_rd

  def __cinit__(self, str filename):
    b_filename = filename.encode('UTF-8')
    at_.atRigSet2(&self._rd, b_filename)
    pass

  def __destoy__(self):
    at_.atRigFree2(self._rd)
    pass

  def atRigidity2(self, x):
    expand = False
    if x.ndim == 0:
      x = x.reshape(1)
      expand = True
    if x.ndim != 1:
      raise ValueError("Invalid dimention: x")
    cdef unsigned int i, n
    n = x.shape[0]
    cdef np.ndarray rig = np.zeros((n, ), dtype=np.double)
    cdef at_.AtPolarVect c_x
    cdef double c_rig
    for i in range(n):
      toPolarVect(x[i], &c_x)
      at_.atRigidity2(self._rd, &c_x, &c_rig)
      rig[i] = c_rig
    if expand:
      return rig[0]
    return rig

def atRigSet2(str filename):
  """\
  Set up Cut-off Rigidity Table in FITS format
  """
  return AtRigData2(filename)

def atRigFree2(AtRigData2 rdp):
  """\
  Free allocated memory by atRigSet2()
  This function exists only API compatibility
  """
  pass

def atRigidity2(AtRigData2 rdp, x):
  """\
  calc Cut-off Rigidity (RIGIDY)

  double atRigidity2(AtRigData2, AtPolarVect)
  double[] atRigidity2(AtRigData2, AtPolarVect[])
  """
  return rdp.atRigidity2(x)


def atGeomagSet(double mjd, int nmax):
  """\
  Set up Table of Geomagnetic Field (GASET)

  void atGeomagSet(duble, int)
  """
  at_.atGeomagSet(mjd, nmax)

def atGeomag(x, xs):
  """\
  calc Geomagnetic Field (GA)

  AtVect atGeomag(atPolarVect, AtVect)
  AtVect[] atGeomag(atPolarVect[], AtVect[])
  """
  expand = False
  if x.ndim == 0 and xs.ndim == 0:
    expand = True
  if x.ndim == 0:
    x = x.reshape(1)
  if xs.ndim == 0:
    xs = xs.reshape(1)
  if x.ndim != 1:
    raise ValueError("Invalid dimention: x")
  if xs.ndim != 1:
    raise ValueError("Invalid dimention: xs")
  if not valid_array_len(x, xs):
    raise ValueError("Invalid array length")
  cdef unsigned int i, n
  n = max(x.shape[0], xs.shape[0])
  cdef np.ndarray field = np.zeros((n, 3), dtype='f8')
  cdef at_.AtPolarVect c_x
  cdef at_.AtVect c_xs, c_field
  for i in range(n):
    toPolarVect(x[min(i, x.shape[0] - 1)], &c_x)
    toVect(xs[min(i, xs.shape[0] - 1)], c_xs)
    at_.atGeomag(&c_x, c_xs, c_field)
    fromVect(c_field, field[i])
  if expand: return field[0]
  return field

def atSun(mjd):
  """\
  calc position of planets, sun and moon (PLANETS, MOON in SOLARSYS)

  ndarray[3] atSun(double)
  ndarray[N,3] atSun(ndarray[N])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_pos
  expand = False
  if is_number(mjd):
    expand = True
    mjd = n2np(mjd)
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  n = mjd.shape[0]
  cdef np.ndarray pos = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    at_.atSun(mjd[i], at_pos)
    fromVect(at_pos, pos[i])
  if expand: return pos[0]
  return pos

def atPlanet(mjd):
  """\
  return pos, size, mag

  AtVect[9], double[9], double[9] atPlanet(double)
  AtVect[][9], double[][9], double[][9] atPlanet(double[])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_pos[9]
  cdef double c_size[9]
  cdef double c_mag[9]
  expand = False
  if is_number(mjd):
    expand = True
    mjd = n2np(mjd)
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  n = mjd.shape[0]
  cdef np.ndarray pos = np.zeros((n, 9, 3), dtype=np.double)
  cdef np.ndarray size = np.zeros((n, 9), dtype=np.double)
  cdef np.ndarray mag = np.zeros((n, 9), dtype=np.double)
  for i in range(n):
    at_.atPlanet(mjd[i], at_pos, c_size, c_mag)
    for j in range(9):
      fromVect(at_pos[j], pos[i][j])
      size[i][j] = c_size[j]
      mag[i][j] = c_mag[j]
  if expand: return pos[0], size[0], mag[0]
  return pos, size, mag

def atMoon(mjd):
  """\
  calc position of the moon from the earth center in TOD coordinate.
  return pos, size, phase, distan

  AtVect, double, double, double atMoon(double)
  AtVect[], double[], double[], double[] atMoon(double[])
  """
  cdef unsigned int n, i
  expand = False
  if is_number(mjd):
    expand = True
    mjd = n2np(mjd)
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  n = mjd.shape[0]
  cdef np.ndarray pos = np.zeros((n, 3), dtype=np.double)
  cdef np.ndarray size = np.zeros((n, ), dtype=np.double)
  cdef np.ndarray phase = np.zeros((n, ), dtype=np.double)
  cdef np.ndarray distan = np.zeros((n, ), dtype=np.double)
  cdef at_.AtVect at_pos
  cdef double c_size, c_phase, c_distan
  for i in range(n):
    at_.atMoon(mjd[i], at_pos, &c_size, &c_phase, &c_distan)
    fromVect(at_pos, pos[i])
    size[i] = c_size
    phase[i] = c_phase
    distan[i] = c_distan
  if expand: return pos[0], size[0], phase[0], distan[0]
  return pos, size, phase, distan


def atSAA(saa_type, np.ndarray x):
  """\
  calc if the point is in the "Brazil Anomaly" (SAA)

  int atSAA(int, AtPolarVect)
   :
  int[] atSAA(int[], AtPolarVect[])
  """
  expand = False
  if is_number(saa_type) and x.ndim == 0:
    expand = True
  if is_number(saa_type):
    saa_type = np.array([int(saa_type)], dtype=np.int_)
  if x.ndim == 0:
    x = x.reshape(1)
  if x.ndim != 1:
    raise ValueError("Invalid dimention: x")
  if valid_array_len(saa_type, x):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(saa_type.shape[0], x.shape[0])
  cdef np.ndarray flag = np.zeros((n, ), dtype=np.int_)
  cdef int c_flag
  cdef at_.AtPolarVect at_x
  for i in range(n):
    toPolarVect(x[min(i, x.shape[0] - 1)], &at_x)
    at_.atSAA(
      saa_type[min(i, saa_type.shape[0] - 1)],
      &at_x,
      &c_flag
    )
    flag[i] = c_flag
  if expand: return flag[0]
  return flag

def atBrazil(lon, lat):
  """\
  int atBrazil(double, double)
  ndarray[N] atBrazil(ndarray[N], double)
  ndarray[N] atBrazil(double, ndarray[N])
  ndarray[N] atBrazil(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(lon) and is_number(lat):
    expand =True
  lon = n2np(lon)
  lat = n2np(lat)
  if lon.ndim != 1:
    raise ValueError("Invalid dimention: lon")
  if lat.ndim != 1:
    raise ValueError("Invalid dimention: lat")
  if not valid_array_len(lon, lat):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(lon.shape[0], lat.shape[0])
  cdef int c_flag
  cdef np.ndarray flag = np.zeros((n, ), dtype=np.int_)
  for i in range(n):
    at_.atBrazil(lon[min(i, lon.shape[0] - 1)], lat[min(i, lat.shape[0] - 1)], &c_flag)
    flag[i] = c_flag
  if expand: return flag[0]
  return flag

def atSISBrazil(lon, lat):
  """\
  int atSISBrazil(double, double)
  ndarray[N] atSISBrazil(ndarray[N], double)
  ndarray[N] atSISBrazil(double, ndarray[N])
  ndarray[N] atSISBrazil(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(lon) and is_number(lat):
    expand =True
  lon = n2np(lon)
  lat = n2np(lat)
  if lon.ndim != 1:
    raise ValueError("Invalid dimention: lon")
  if lat.ndim != 1:
    raise ValueError("Invalid dimention: lat")
  if not valid_array_len(lon, lat):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(lon.shape[0], lat.shape[0])
  cdef int c_flag
  cdef np.ndarray flag = np.zeros((n, ), dtype=np.int_)
  for i in range(n):
    at_.atSISBrazil(lon[min(i, lon.shape[0] - 1)], lat[min(i, lat.shape[0] - 1)], &c_flag)
    flag[i] = c_flag
  if expand: return flag[0]
  return flag

def atSTTBrazil(lon, lat):
  """\
  int atSTTBrazil(double, double)
  ndarray[N] atSTTBrazil(ndarray[N], double)
  ndarray[N] atSTTBrazil(double, ndarray[N])
  ndarray[N] atSTTBrazil(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(lon) and is_number(lat):
    expand =True
  lon = n2np(lon)
  lat = n2np(lat)
  if lon.ndim != 1:
    raise ValueError("Invalid dimention: lon")
  if lat.ndim != 1:
    raise ValueError("Invalid dimention: lat")
  if not valid_array_len(lon, lat):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(lon.shape[0], lat.shape[0])
  cdef int c_flag
  cdef np.ndarray flag = np.zeros((n, ), dtype=np.int_)
  for i in range(n):
    at_.atSTTBrazil(lon[min(i, lon.shape[0] - 1)], lat[min(i, lat.shape[0] - 1)], &c_flag)
    flag[i] = c_flag
  if expand: return flag[0]
  return flag

def atHXDBrazil(lon, lat):
  """\
  int atHXDBrazil(double, double)
  ndarray[N] atHXDBrazil(ndarray[N], double)
  ndarray[N] atHXDBrazil(double, ndarray[N])
  ndarray[N] atHXDBrazil(ndarray[N], ndarray[N])
  """
  expand = False
  if is_number(lon) and is_number(lat):
    expand =True
  lon = n2np(lon)
  lat = n2np(lat)
  if lon.ndim != 1:
    raise ValueError("Invalid dimention: lon")
  if lat.ndim != 1:
    raise ValueError("Invalid dimention: lat")
  if not valid_array_len(lon, lat):
    raise ValueError("Invalid array length")
  cdef unsigned int n, i
  n = max(lon.shape[0], lat.shape[0])
  cdef int c_flag
  cdef np.ndarray flag = np.zeros((n, ), dtype=np.int_)
  for i in range(n):
    at_.atHXDBrazil(lon[min(i, lon.shape[0] - 1)], lat[min(i, lat.shape[0] - 1)], &c_flag)
    flag[i] = c_flag
  if expand: return flag[0]
  return flag



##############################
# New for Astro-D
##############################

# ERROR: Symbol not found: _atBarycentric
# def atBarycentric(mjd, np.ndarray x):
#   """\
#   convert geocentric vector to Barycentric
#
#   ndarray[3] atBarycentric(double, ndarray[3])
#   ndarray[N,3] atBarycentric(ndarray[N], ndarray[3])
#   ndarray[N,3] atBarycentric(double, ndarray[N,3])
#   ndarray[N,3] atBarycentric(ndarray[N], ndarray[N,3])
#   """
#   cdef unsigned int n, i
#   cdef at_.AtVect at_x, at_y
#   cdef np.ndarray y
#   expand = False
#   if is_number(mjd) and x.ndim == 1:
#     expand = True
#   mjd = n2np(mjd)
#   if x.ndim == 1:
#     x = x.reshape((1, 3))
#   n = max(mjd.shape[0], x.shape[0])
#   y = np.zeros((n, 3), dtype=np.double)
#   for i in range(n):
#     toVect(x[min(i, x.shape[0] -  1)], at_x)
#     at_.atBarycentric(mjd[min(i, mjd.shape[0] - 1)], at_x, at_y)
#     fromVect(at_y, y[i])
#   if expand: return y[0]
#   return y

# ERROR: Symbol not found: _atTDT
# def atTDT(mjd):
#   """\
#   convert MJD to TDT (terrestial dynamical time)
#
#   double atTDT(double)
#   ndarray[N] atTDT(ndarray[N])
#   """
#   cdef unsigned int n, i
#   cdef double c_tdt
#   cdef np.ndarray tdt
#   expand = False
#   if is_number(mjd):
#     expand = True
#     mjd = n2np(mjd)
#   if mjd.ndim != 1:
#     raise ValueError("Invalid dimention: mjd")
#   n = mjd.shape[0]
#   tdt = np.zeros((n, ), dtype=np.double)
#   for i in range(n):
#     at_.atTDT(mjd[i], &c_tdt)
#     tdt[i] = c_tdt
#   if expand: return tdt[0]
#   return tdt


##############################
# since version 1.8
##############################

def atEarthElev(np.ndarray vSat, np.ndarray nvTgt, np.ndarray vSun):
  """\
  examine (day|night) earth elevation of specified direction

  int, ndarray[3] atEarthElev(np.ndarray[3], np.ndarry[3] , np.ndarray[3])
  ndarray[N], ndarray[N,3] atEarthElev(np.ndarray[N,3], np.ndarry[3] , np.ndarray[3])
   :
  ndarray[N], ndarray[N,3] atEarthElev(np.ndarray[N,3], np.ndarry[N,3] , np.ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_vSat, at_nvTgt, at_vSun
  cdef int c_occult
  cdef double c_elv[3]
  expand = False
  if vSat.ndim == 1 and nvTgt.ndim == 1 and vSun.ndim == 1:
    expand = True
  if vSat.ndim == 1:
    vSat = vSat.reshape((1, 3))
  if nvTgt.ndim == 1:
    nvTgt = nvTgt.reshape((1, 3))
  if vSun.ndim == 1:
    vSun = vSun.reshape((1, 3))
  if vSat.ndim != 2:
    raise ValueError("Invalid dimention: vSat")
  if nvTgt.ndim != 2:
    raise ValueError("Invalid dimention: nvTgt")
  if vSun.ndim != 2:
    raise ValueError("Invalid dimention: vSun")
  if not valid_array_len(vSat, nvTgt, vSun):
    raise ValueError("Invalid array length")
  n = max(vSat.shape[0], nvTgt.shape[0], vSun.shape[0])
  cdef np.ndarray occult = np.zeros((n, ), dtype=np.int_)
  cdef np.ndarray elv = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toVect(vSat[min(i, vSat.shape[0] - 1)], at_vSat)
    toVect(nvTgt[min(i, nvTgt.shape[0] - 1)], at_nvTgt)
    toVect(vSun[min(i, vSun.shape[0] - 1)], at_vSun)
    at_.atEarthElev(at_vSat, at_nvTgt, at_vSun, &c_occult, c_elv)
    occult[i] = c_occult
    elv[i][0] = c_elv[0]
    elv[i][1] = c_elv[1]
    elv[i][2] = c_elv[2]
  if expand: return occult[0], elv[0]
  return occult, elv

def atAberration(mjd, np.ndarray x0):
  """\
  correct aberration

  ndarray[3] atAberration(double, ndarray[3])
  ndarray[N,3] atAberration(ndarray[N], ndarray[3])
  ndarray[N,3] atAberration(double, ndarray[N,3])
  ndarray[N,3] atAberration(ndarray[N], ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x0, at_x
  expand = False
  if is_number(mjd) and x0.ndim == 1:
    expand = True
  mjd = n2np(mjd)
  if x0.ndim == 1:
    x0 = x0.reshape((1, 3))
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if x0.ndim != 2:
    raise ValueError("Invalid dimention: x0")
  if not valid_array_len(mjd, x0):
    raise ValueError("Invalid array length")
  n = max(mjd.shape[0], x0.shape[0])
  cdef np.ndarray x = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toVect(x0[min(i, x0.shape[0] - 1)], at_x0)
    at_.atAberration(mjd[min(i, mjd.shape[0] - 1)], at_x0, at_x)
    fromVect(at_x, x[i])
  if expand: return x[0]
  return x

def atInvAberration(mjd, np.ndarray x0):
  """\
  ndarray[3] atInvAberration(double, ndarray[3])
  ndarray[N,3] atInvAberration(ndarray[N], ndarray[3])
  ndarray[N,3] atInvAberration(double, ndarray[N,3])
  ndarray[N,3] atInvAberration(ndarray[N], ndarray[N,3])
  """
  cdef unsigned int n, i
  cdef at_.AtVect at_x0, at_x
  expand = False
  if is_number(mjd) and x0.ndim == 1:
    expand = True
  mjd = n2np(mjd)
  if x0.ndim == 1:
    x0 = x0.reshape((1, 3))
  if mjd.ndim != 1:
    raise ValueError("Invalid dimention: mjd")
  if x0.ndim != 2:
    raise ValueError("Invalid dimention: x0")
  if not valid_array_len(mjd, x0):
    raise ValueError("Invalid array length")
  n = max(mjd.shape[0], x0.shape[0])
  cdef np.ndarray x = np.zeros((n, 3), dtype=np.double)
  for i in range(n):
    toVect(x0[min(i, x0.shape[0] - 1)], at_x0)
    at_.atInvAberration(mjd[min(i, mjd.shape[0] - 1)], at_x0, at_x)
    fromVect(at_x, x[i])
  if expand: return x[0]
  return x

def atJ2000toB1950(np.ndarray pos):
  """\
  ndaray[2] atJ2000toB1950(ndarray[2])
  ndaray[N,2] atJ2000toB1950(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atJ2000toB1950(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atB1950toJ2000(np.ndarray pos):
  """\
  ndaray[2] atB1950toJ2000(ndarray[2])
  ndaray[N,2] atB1950toJ2000(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atB1950toJ2000(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atJ2000toGal(np.ndarray pos):
  """\
  ndaray[2] atJ2000toGal(ndarray[2])
  ndaray[N,2] atJ2000toGal(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atJ2000toGal(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atGaltoJ2000(np.ndarray pos):
  """\
  ndaray[2] atGaltoJ2000(ndarray[2])
  ndaray[N,2] atGaltoJ2000(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atGaltoJ2000(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atGaltoB1950(np.ndarray pos):
  """\
  ndaray[2] atGaltoB1950(ndarray[2])
  ndaray[N,2] atGaltoB1950(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atGaltoB1950(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atB1950toGal(np.ndarray pos):
  """\
  ndaray[2] atB1950toGal(ndarray[2])
  ndaray[N,2] atB1950toGal(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double lo, la
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atB1950toGal(pos[i][0], pos[i][1], &lo, &la)
    pos2[i][0] = lo
    pos2[i][1] = la
  if expand: return pos2[0]
  return pos2

def atJ2000ToEcliptic(np.ndarray pos):
  """\
  ndaray[2] atJ2000ToEcliptic(ndarray[2])
  ndaray[N,2] atJ2000ToEcliptic(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double l, b
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atJ2000ToEcliptic(pos[i][0], pos[i][1], &l, &b)
    pos2[i][0] = l
    pos2[i][1] = b
  if expand: return pos2[0]
  return pos2

def atEclipticToJ2000(np.ndarray pos):
  """\
  ndaray[2] atEclipticToJ2000(ndarray[2])
  ndaray[N,2] atEclipticToJ2000(ndarray[N,2])
  """
  cdef unsigned int n, i
  cdef double alpha, delta
  expand = False
  if pos.ndim == 1:
    expand = True
    pos = pos.reshape((1, 2))
  if pos.ndim != 2:
    raise ValueError("Invalid dimention: pos")
  n = pos.shape[0]
  cdef np.ndarray pos2 = np.zeros((n, 2), dtype=np.double)
  for i in range(n):
    at_.atEclipticToJ2000(pos[i][0], pos[i][1], &alpha, &delta)
    pos2[i][0] = alpha
    pos2[i][1] = delta
  if expand: return pos2[0]
  return pos2

def atRadianToRA(radian):
  """\
  covert degree to Right Ascension in hr,min,s.

  AtRightAscension atRadianToRA(double)
  AtRightAscension[] atRadianToRA(double[])
  """
  cdef unsigned int n, i
  cdef at_.AtRightAscension at_ra
  expand = False
  if is_number(radian):
    expand = True
    radian = n2np(radian)
  if radian.ndim != 1:
    raise ValueError("Invalid dimention: radian")
  n = radian.shape[0]
  cdef np.ndarray ra = np.zeros((n, ), dtype=AtRightAscension)
  for i in range(n):
    at_.atRadianToRA(radian[i], &at_ra)
    fromRA(&at_ra, ra[i])
  if expand: return ra[0]
  return ra

# void atDegToRA(
# 	double deg,		/* input: degree */
# 	AtRightAscension *ra);	/* output: Right Ascension */
def atDegToRA(deg):
  """\
  covert degree to Right Ascension in hr,min,s.

  AtRightAscension atDegToRA(double)
   :
  AtRightAscension[] atDegToRA(double[])
  """
  cdef unsigned int n, i
  cdef at_.AtRightAscension at_ra
  expand = False
  if is_number(deg):
    expand = True
    deg = n2np(deg)
  if deg.ndim != 1:
    raise ValueError("Invalid dimention: radian")
  n = deg.shape[0]
  cdef np.ndarray ra = np.zeros((n, ), dtype=AtRightAscension)
  for i in range(n):
    at_.atDegToRA(deg[i], &at_ra)
    fromRA(&at_ra, ra[i])
  if expand: return ra[0]
  return ra

  # void atRadianToDec(
  # 	double radian,		/* input: radian */
  # 	AtDeclination *dec);	/* output: Declination */
def atRadianToDec(radian):
  """\
  covert radian to Declination in deg,min,s.

  AtDeclination atRadianToDec(double)
  AtDeclination[] atRadianToDec(double[])
  """
  cdef unsigned int n, i
  cdef at_.AtDeclination at_dec
  expand = False
  if is_number(radian):
    expand = True
    radian = n2np(radian)
  if radian.ndim != 1:
    raise ValueError("Invalid dimention: radian")
  n = radian.shape[0]
  cdef np.ndarray dec = np.zeros((n, ), dtype=AtDeclination)
  for i in range(n):
    at_.atRadianToDec(radian[i], &at_dec)
    fromDec(&at_dec, dec[i])
  if expand: return dec[0]
  return dec

# void atDegToDec(
# 	double deg,		/* input: degree */
# 	AtDeclination *dec);	/* output: Declination */
def atDegToDec(deg):
  """\
  covert degree to Declination in deg,min,s.

  AtDeclination atDegToDec(double)
  AtDeclination[] atDegToDec(double[])
  """
  cdef unsigned int n, i
  cdef at_.AtDeclination at_dec
  expand = False
  if is_number(deg):
    expand = True
    deg = n2np(deg)
  if deg.ndim != 1:
    raise ValueError("Invalid dimention: deg")
  n = deg.shape[0]
  cdef np.ndarray dec = np.zeros((n, ), dtype=AtDeclination)
  for i in range(n):
    at_.atDegToDec(deg[i], &at_dec)
    fromDec(&at_dec, dec[i])
  if expand: return dec[0]
  return dec

# double atParseRAToRadian(	/* return: Right Ascension in radian */
# 	char *expression);	/* input: number (degree) or 00h00m00.0s */
def atParseRAToRadian(expression):
  """\
  double atParseRAToRadian(str)
  ndarray[N] atParseRAToRadian(str[])
  """
  cdef unsigned int n, i
  expand = False
  if isinstance(expression, str):
    expand = True
    expression = [expression]
  n = len(expression)
  cdef np.ndarray radian = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    exp = expression[i].encode()
    radian[i] = at_.atParseRAToRadian(exp)
  if expand: return radian[0]
  return radian

# double atParseDecToRadian(	/* return: Declination in radian */
# 	char *expression);	/* input: number (degree) or +00d00m00.0s */
def atParseDecToRadian(expression):
  """\
  double atParseDecToRadian(str)
  double[] atParseDecToRadian(str[])
  """
  cdef unsigned int n, i
  cdef char *c_expression
  expand = False
  if isinstance(expression, str):
    expand = True
    expression = [expression]
  n = len(expression)
  cdef np.ndarray radian = np.zeros((n, ), dtype=np.double)
  for i in range(n):
    exp = expression[i].encode()
    c_expression = exp
    radian[i] = at_.atParseDecToRadian(c_expression)
  if expand: return radian[0]
  return radian


##############################
# since version 2.5
##############################

# int atQuatToEuler(
# 	AtQuat q,		/* input: quaternion */
# 	AtEulerAng *ea);	/* output: z-y-z Euler Angle (radian) */
def atQuatToEuler(np.ndarray q):
  """\
  convert quaternion parameter set to Euler angles

  AtEulerAng atQuatToEuler(AtQuat)
  AtEulerAng[] atQuatToEuler(AtQuat[])
  """
  cdef unsigned int n, i
  expand = False
  if q.ndim == 1:
    expand = True
    q = q.reshape((1, 4))
  if q.ndim != 2:
    raise ValueError("Invalid dimention: q")
  n = q.shape[0]
  cdef np.ndarray ea = np.zeros((n, ), dtype=AtEulerAng)
  cdef at_.AtEulerAng at_ea
  cdef at_.AtQuat at_q
  for i in range(n):
    toQuat(q[i], at_q)
    at_.atQuatToEuler(at_q, &at_ea)
    fromEulerAng(&at_ea, ea[i])
  if expand: return ea[0]
  return ea

# int atEulerToQuat(
# 	AtEulerAng *ea,		/* input: z-y-z Euler Angle (radian) */
# 	AtQuat q);		/* output: quaternion */
def atEulerToQuat(np.ndarray ea):
  """\
  convert Euler angles to quaternion parameter set

  AtQuat atEulerToQuat(AtEulerAng)
   :
  AtQuat[] atEulerToQuat(AtEulerAng[])
  """
  cdef unsigned int n, i
  expand = False
  if ea.ndim == 0:
    expand = True
    ea = ea.reshape(1)
  if ea.ndim != 1:
    raise ValueError("Invalid dimention: q")
  n = ea.shape[0]
  cdef np.ndarray q = np.zeros((n, 4), dtype=np.double)
  cdef at_.AtQuat at_q
  cdef at_.AtEulerAng at_ea
  for i in range(n):
    toEulerAng(ea[i], &at_ea)
    at_.atEulerToQuat(&at_ea, at_q)
    fromQuat(at_q, q[i])
  if expand: return q[0]
  return q

# int
# atInterpolateQuat(
# 	double t0, AtQuat q0,	/* input: q-parameter q0 at time t0 */
# 	double t1, AtQuat q1,	/* input: q-parameter q0 at time t0 */
# 	double t,  AtQuat q	/* input: time t
# 				   output: interpolated q-parameter q */
# );
def atInterpolateQuat(t0, np.ndarray q0, t1, np.ndarray q1, t):
  """\
  interpolation between two sets of q-parameters

  AtQuat atInterpolateQuat(double, AtQuat, double, AtQuat, double)
   :
  AtQuat[] atInterpolateQuat(double[], AtQuat[], double[], AtQuat[], double[])
  """
  cdef unsigned int n, i
  cdef at_.AtQuat at_q0, at_q1, at_q
  expand = False
  if is_number(t0) and q0.ndim == 1 and is_number(t1) and q1.ndim == 1 and is_number(t):
    expand = True
  t0 = n2np(t0)
  if q0.ndim == 1:
    q0 = q0.reshape((1, 4))
  t1 = n2np(t1)
  if q1.ndim == 1:
    q1 = q1.reshape((1, 4))
  t = n2np(t)
  if t0.ndim != 1:
    raise ValueError("Invalid dimention: t0")
  if q0.ndim != 2:
    raise ValueError("Invalid dimention: q0")
  if t1.ndim != 1:
    raise ValueError("Invalid dimention: t1")
  if q1.ndim != 2:
    raise ValueError("Invalid dimention: q1")
  if t.ndim != 1:
    raise ValueError("Invalid dimention: t")
  if not valid_array_len(t0, q0, t1, q1, t):
    raise ValueError("Invalid array length")
  n = max(t0.shape[0], q0.shape[0], t1.shape[0], q1.shape[0], t.shape[0])
  cdef np.ndarray q = np.zeros((n, 4), dtype=np.double)
  for i in range(n):
    toQuat(q0[min(i, q0.shape[0] - 1)], at_q0)
    toQuat(q1[min(i, q1.shape[0] - 1)], at_q1)
    at_.atInterpolateQuat(
      t0[min(i, t0.shape[0] - 1)],
      at_q0,
      t1[min(i, t1.shape[0] - 1)],
      at_q1,
      t[min(i, t.shape[0] - 1)],
      at_q
    )
    fromQuat(at_q, q[i])
  if expand: return q[0]
  return q


# int
# atInterpolateEuler(
# 	double t0, AtEulerAng *ea0,	/* input: q-parameter q0 at time t0 */
# 	double t1, AtEulerAng *ea1,	/* input: q-parameter q0 at time t0 */
# 	double t,  AtEulerAng *ea	/* input: time t
# 					   output: interpolated Euler angles */
def atInterpolateEuler(t0, np.ndarray ea0, t1, np.ndarray ea1, t):
  """\
  interpolation between two sets of Euler angles

  AtEulerAng atInterpolateEuler(double, AtEulerAng, double, AtEulerAng, double)
   :
  AtEulerAng[] atInterpolateEuler(double[], AtEulerAng[], double[], AtEulerAng[], double[])
  """
  cdef unsigned int n, i
  cdef at_.AtEulerAng at_ea0, at_ea1, at_ea
  expand = False
  if is_number(t0) and ea0.ndim == 0 and is_number(t1) and ea1.ndim == 0 and is_number(t):
    expand = True
  t0 = n2np(t0)
  if ea0.ndim == 0:
    ea0 = ea0.reshape(1)
  t1 = n2np(t1)
  if ea1.ndim == 0:
    ea1 = ea1.reshape(1)
  t = n2np(t)
  if t0.ndim != 1:
    raise ValueError("Invalid dimention: t0")
  if ea0.ndim != 1:
    raise ValueError("Invalid dimention: ea0")
  if t1.ndim != 1:
    raise ValueError("Invalid dimention: t1")
  if ea1.ndim != 1:
    raise ValueError("Invalid dimention: ea1")
  if t.ndim != 1:
    raise ValueError("Invalid dimention: t")
  if not valid_array_len(t0, ea0, t1, ea1, t):
    raise ValueError("Invalid array length")
  n = max(t0.shape[0], ea0.shape[0], t1.shape[0], ea1.shape[0], t.shape[0])
  cdef np.ndarray ea = np.zeros((n, ), dtype=AtEulerAng)
  for i in range(n):
    toEulerAng(ea0[min(i, ea0.shape[0] - 1)], &at_ea0)
    toEulerAng(ea1[min(i, ea1.shape[0] - 1)], &at_ea1)
    at_.atInterpolateEuler(
      t0[min(i, t0.shape[0] - 1)],
      &at_ea0,
      t1[min(i, t1.shape[0] - 1)],
      &at_ea1,
      t[min(i, t.shape[0] - 1)],
      &at_ea
    )
    fromEulerAng(&at_ea, ea[i])
  if expand: return ea[0]
  return ea

############################################################
# pyat original
############################################################
