import unittest
import os
import numpy as np
import atFunctions as at
from numpy.testing import (
    assert_array_equal,
    assert_almost_equal,
    assert_array_almost_equal
)

class TestAtFunctions(unittest.TestCase):
    def test_n2np(self):
        """\
        test for n2np()
        """
        val = at.n2np(5)
        self.assertTrue(isinstance(val, np.ndarray))
        self.assertEqual(val.ndim, 1)
        self.assertEqual(val[0], 5)
        val = at.n2np(np.array([5]))
        self.assertTrue(isinstance(val, np.ndarray))
        self.assertEqual(val.ndim, 1)
        self.assertEqual(val[0], 5)

    def test_array_mul(self):
        """\
        test for array_mul()
        """
        val = at.array_mul([1, 2, 3, 4, 5])
        self.assertEqual(val, 120)
        val = at.array_mul(np.array([1, 2, 3, 4, 5]))
        self.assertEqual(val, 120)

    def test_is_number(self):
        """\
        test for is_number()
        """
        a = 5.0
        b = np.array([5.0])
        self.assertTrue(at.is_number(a))
        self.assertFalse(at.is_number(b))

    def test_valid_array_len(self):
        """\
        test for valid_array_len()
        """
        a = np.array([1])
        b = np.array([100])
        c = np.array([1, 2, 3])
        d = np.array([1, 3, 5])
        e = np.array([1, 2, 3, 4, 5])
        f = np.array([1, 3, 5, 7, 9])
        self.assertTrue(at.valid_array_len(a, b))
        self.assertTrue(at.valid_array_len(c, d))
        self.assertTrue(at.valid_array_len(e, f))
        self.assertTrue(at.valid_array_len(a, c))
        self.assertTrue(at.valid_array_len(a, e))
        self.assertTrue(at.valid_array_len(a, c, d))
        self.assertTrue(at.valid_array_len(a, b, c, d))
        self.assertTrue(at.valid_array_len(a, b, e))
        self.assertTrue(at.valid_array_len(a, b, e, f))
        self.assertFalse(at.valid_array_len(c, e))
        self.assertFalse(at.valid_array_len(a, c, e))
        self.assertFalse(at.valid_array_len(c, d, e))
        self.assertFalse(at.valid_array_len(a, c, d, e))
        self.assertFalse(at.valid_array_len(a, c, e, f))

    def test_atRAToRadian(self):
        """\
        test for atRaToRadian / atRadianToRA
        """
        ra = np.array((1, 30, 0), dtype=at.AtRightAscension)
        radian = at.atRAToRadian(ra)
        assert_almost_equal(radian, 0.39269908169872414)
        ra2 = at.atRadianToRA(radian)
        assert_array_almost_equal(
            [ra2['hour'], ra2['min'], ra2['sec']],
            [ra['hour'], ra['min'], ra['sec']])
        ra = np.array([(1, 30, 0), (12, 20, 10)], dtype=at.AtRightAscension)
        radian = at.atRAToRadian(ra)
        assert_array_almost_equal(radian, [0.39269908169872414, 3.229586336711174], 9)
        ra2 = at.atRadianToRA(radian)
        assert_array_almost_equal(
            [
                [ra2[0]['hour'], ra2[0]['min'], ra2[0]['sec']],
                [ra2[1]['hour'], ra2[0]['min'], ra2[0]['sec']],
            ],
            [
                [ra[0]['hour'], ra[0]['min'], ra[0]['sec']],
                [ra[1]['hour'], ra[0]['min'], ra[0]['sec']],
            ])

    def test_atDecToRadian(self):
        """\
        test for atDecToRadian / atRadianToDec
        """
        dec = np.array((1, 40, 30, 5), dtype=at.AtDeclination)
        radian = at.atDecToRadian(dec)
        assert_almost_equal(radian, 0.7068825877417589)
        dec2 = at.atRadianToDec(radian)
        assert_array_almost_equal(
            [dec2['sign'], dec2['deg'], dec2['min'], dec2['sec']],
            [dec['sign'],  dec['deg'],  dec['min'],  dec['sec']]
        )
        dec = np.array((-1, 40, 30, 5), dtype=at.AtDeclination)
        radian = at.atDecToRadian(dec)
        assert_almost_equal(radian, -0.7068825877417589)
        dec = np.array([(1, 40, 30, 5), (-1, 22, 51, 13)], dtype=at.AtDeclination)
        radian = at.atDecToRadian(dec)
        assert_array_almost_equal(radian, [0.7068825877417589, -0.3988707598592486], 9)
        dec2 = at.atRadianToDec(radian)
        assert_array_almost_equal(
            [
                [dec2[0]['sign'], dec2[0]['deg'], dec2[0]['min'], dec2[0]['sec']],
                [dec2[1]['sign'], dec2[1]['deg'], dec2[1]['min'], dec2[1]['sec']],
            ],
            [
                [dec[0]['sign'], dec[0]['deg'], dec[0]['min'], dec[0]['sec']],
                [dec[1]['sign'], dec[1]['deg'], dec[1]['min'], dec[1]['sec']],
            ],
        )

    def test_atVectToPolDeg(self):
        """\
        test for atVectToPolDeg and atPoDegToVect
        """
        x = np.array([1.0, 4.0, 3.0], dtype=np.double)
        r, alpha, delta = at.atVectToPolDeg(x)
        assert_array_almost_equal(
            [r, alpha, delta],
            [5.0990195135927845, 75.96375653207353, 36.039893430303856],
            9
        )
        x2 = at.atPolDegToVect(r, alpha, delta)
        assert_array_almost_equal(x, x2, 9)
        x = np.array([[3.0, 2.5, 1.8], [4.9, 2.2, 10.0]], dtype=np.double)
        y = at.atVectToPolDeg(x)
        assert_array_almost_equal(
            [
                [y[0][0], y[1][0], y[2][0]],
                [y[0][1], y[1][1], y[2][1]],
            ],
            [
                [4.3, 39.80557109226519, 24.746524586937863],
                [11.351211389098522, 24.17910714668918, 61.75877805190976],
            ],
            9
        )
        x2 = at.atPolDegToVect(*y)
        assert_array_almost_equal(x, x2, 9)

    def test_atRotVect(self):
        """\
        test for atRotVect
        """
        rm = at.atSetRotMat(np.array([1.68535245, 1.53415867, 0.22952997], dtype=np.double), 1.96789251)
        x = np.array([ 0.67271918,  0.17667265, -0.49965835], dtype=np.double)
        y = at.atRotVect(rm, x)
        assert_array_almost_equal(y, [0.63945708, 0.05360649, 0.56713647])
        rm = at.atSetRotMat(np.array([
            [-1.19103241, -1.31812744, -0.532126  ],
            [ 0.39742137,  0.78295887, -0.69146437]], dtype=np.double),
            np.array([ 0.07531121, -1.1931914 ], dtype=np.double))
        x = np.array([
            [ 1.50892821, -1.31840297,  0.96494652],
            [ 1.72373185,  0.78995717,  0.50694645]], dtype=np.double)
        y = at.atRotVect(rm, x)
        assert_array_almost_equal(y, [
            [ 1.58528053, -1.32809524,  0.81805941],
            [ 1.61138625, -0.49051312, -1.00752655]])

    def test_atSetRotMat(self):
        """\
        test for atSetRotMat
        """
        x = np.array([1.0, 0.0, 0.0], dtype=np.double)
        roll = at.PI / 2.0
        rm = at.atSetRotMat(x, roll)
        assert_array_almost_equal(rm, [
            [1.0, 0.0, 0.0],
            [0.0, 0.0, 1.0],
            [0.0, -1.0, 0.0],
        ], 9)
        x = np.array([
            [1.0, 0.0, 0.0],
            [0.0, 1.0, 0.0],
        ], dtype=np.double)
        roll = np.array([at.PI / 2.0, at.PI / 4.0], dtype=np.double)
        rm = at.atSetRotMat(x, roll)
        assert_array_almost_equal(rm, [
            [
                [ 1.00000000e+00,  0.00000000e+00,  0.00000000e+00],
                [ 0.00000000e+00,  6.12323400e-17,  1.00000000e+00],
                [ 0.00000000e+00, -1.00000000e+00,  6.12323400e-17],
            ],
            [
                [ 7.07106781e-01,  0.00000000e+00, -7.07106781e-01],
                [ 0.00000000e+00,  1.00000000e+00,  0.00000000e+00],
                [ 7.07106781e-01,  0.00000000e+00,  7.07106781e-01],
            ],
        ])

    def test_atSetRotMatZX(self):
        """\
        test for atSetRotMatZX
        """
        z = np.array([1.0, 3.0, 5.0], dtype=np.double)
        x = np.array([4.0, 2.0, 1.0], dtype=np.double)
        rm = at.atSetRotMatZX(z, x)
        assert_array_almost_equal(rm, [
            [ 0.93560149,  0.1871203 , -0.29939248],
            [-0.30996521,  0.84133414, -0.44280744],
            [ 0.16903085,  0.50709255,  0.84515425],
        ])
        z = np.array([[2.5, 3.0, 5.3], [9.2, 3.5, 8.5]], dtype=np.double)
        x = np.array([[9.3, 4.6, 3.3], [3.9, 2.5, 5.2]], dtype=np.double)
        rm =at.atSetRotMatZX(z, x)
        assert_array_almost_equal(rm, [
            [
                [ 0.87112139,  0.11674526, -0.47698854],
                [-0.31135019,  0.88244557, -0.35263419],
                [ 0.37974805,  0.45569766,  0.80506587],
            ],
            [
                [-0.68543428,  0.48690234,  0.54139261],
                [-0.17252823, -0.83096384,  0.52889802],
                [ 0.70739936,  0.26911932,  0.6535755 ],
            ]
        ])

    def test_atInvRotMat(self):
        """\
        test atInvRotMat
        """
        x = np.array([1.3, 5.0, 2.2], dtype=np.double)
        roll = at.PI / 4.0
        rm = at.atSetRotMat(x, roll)
        rm2 = at.atInvRotMat(rm)
        rm3 = at.atInvRotMat(rm2)
        assert_array_almost_equal(rm, rm3)
        x = np.array([[5.0, 3.0, 1.0], [2.0, 4.0, 1.0]], dtype=np.double)
        roll = np.array([at.PI / 3.0, at.PI /4.0], dtype=np.double)
        rm = at.atSetRotMat(x, roll)
        rm2 = at.atInvRotMat(rm)
        rm3 = at.atInvRotMat(rm2)
        assert_array_almost_equal(rm, rm3)

    def test_atNormVect(self):
        """\
        test atNormVect
        """
        x = np.array([4.0, 2.0, 3.0], dtype=np.double)
        y = at.atNormVect(x)
        assert_array_almost_equal(y, [0.74278135, 0.37139068, 0.55708601])
        x = np.array([[2.5, 5.9, 8.1], [9.2, 4.4, 7.1]], dtype=np.double)
        y = at.atNormVect(x)
        assert_array_almost_equal(y, [
            [0.24205768, 0.57125612, 0.78426687],
            [0.74037222, 0.35409106, 0.57137421]
        ])

    def test_atNorm(self):
        """\
        test for atNorm
        """
        x = np.array([ 0.04750462,  1.09470412, -0.0687464 ])
        y = at.atNorm(x)
        assert_almost_equal(y, 1.097888822595111)
        x = np.array([
            [ 1.3980351 , -1.20447002,  1.27073108],
            [ 0.38209843, -0.55582178, -2.39374945]
        ])
        y = at.atNorm(x)
        assert_almost_equal(y, [2.24053736, 2.48696069])

    def test_atRMProd(self):
        """\
        test atRMProd
        """
        rm0 = np.array([
            [-1.55662022, -0.10351657,  0.45652284],
            [ 0.19371174, -1.13609301, -0.07698105],
            [-1.30230128, -0.66853609, -1.23815925],
        ], dtype=np.double)
        rm1 = np.array([
            [ 0.55451637,  0.55440807, -0.52979441],
            [ 1.67862489,  0.4015806 ,  0.2299147 ],
            [-0.95143057, -1.45648761,  0.42811989],
        ], dtype=np.double)
        rm2 = at.atRMProd(rm0, rm1)
        assert_array_almost_equal(rm2, [
            [-0.0658241 , -0.33307408,  0.86644032],
            [-2.83460878, -0.78370468,  0.45074549],
            [ 0.64133623,  1.46698062, -0.85230844]
        ])

    def test_atRMCheck(self):
        """\
        test for atRMCheck
        """
        rm = np.array([
            [ 0.26414758, -0.69096561, -0.68520126],
            [-0.1604683 , -2.09629037, -0.99785689],
            [ 0.64043501, -0.13870299, -0.04509599],
        ], dtype=np.double)
        rm2 = at.atRMCheck(rm)
        assert_array_almost_equal(rm2, [
            [ 0.88637929, -0.43818091, -0.13301773],
            [-0.44971389, -0.77831172, -0.43284265],
            [ 0.08633446,  0.44451416, -0.88899831],
        ])

    def test_atRMToEuler(self):
        """\
        test for atRMToEuler / atEulerToRM
        """
        ea = np.array((at.PI / 6.0, at.PI / 4.0, at.PI / 3.0), dtype=at.AtEulerAng)
        rm = at.atEulerToRM(ea)
        ea2 = at.atRMToEuler(rm)
        assert_array_almost_equal(
            [ea['phi'], ea['theta'], ea['psi']],
            [ea2['phi'], ea2['theta'], ea2['psi']]
        )

    def atRMToQuat(self):
        """\
        test for atRMToQuat / atQuatToRM
        """
        q = np.array([-0.33270991, -0.45489815, -0.04799774,  0.4366393 ], np.double)
        rm = at.atQuatToRM(q)
        q2 = at.atRMToQuat(rm)
        assert_array_almost_equal(q, q2)
        q = np.array([
            [-1.45372336e+00,  9.76212305e-01, -1.67196767e+00, -1.86691819e-01],
            [-1.03105630e+00, -6.66048757e-01, -1.14569142e-03, 9.69218320e-01]
        ])
        rm = at.atQuatToRM(q)
        q2 = at.atRMToQuat(rm)
        assert_array_almost_equal(q, q2)

    def test_atQuatProd(self):
        """\
        test for atQuatProd
        """
        q0 = np.array([-1.51961738,  0.23515613,  0.94964713, -2.23841655], dtype=np.double)
        q1 = np.array([-0.54986558, -0.31352514,  1.09792179,  1.26974969], dtype=np.double)
        q2 = at.atQuatProd(q0, q1)
        assert_array_almost_equal(q2, [-0.14278419,  2.14663205, -0.64604964, -4.64672493])
        q0 = np.array([
            [ 0.05773771,  1.24436886, -1.19266735, -0.98934436],
            [-1.08749452, -0.10810819,  1.15238314,  0.63175863]
        ], dtype=np.double)
        q1 = np.array([
            [-1.22277976,  1.2822201 ,  0.82330284,  0.27805051],
            [ 0.84777547,  0.02639238, -0.0832676 , -0.52335321]
        ], dtype=np.double)
        q2 = at.atQuatProd(q0, q1)
        assert_array_almost_equal(q2, [
            [ 3.77955872,  0.48827405,  0.44946972, -0.81811555],
            [ 1.08332099,  0.95966148, -0.59275854,  0.69012768],
        ])

    def test_atCopyVect(self):
        """\
        test for atCopyVect
        """
        x = np.array([ 0.79539486,  0.43061085, -0.65511234], dtype=np.double)
        y = at.atCopyVect(x)
        assert_array_equal(x, y)
        # check for y is not reference of x
        y[0] = y[0] + 1.0
        self.assertNotEqual(x[0], y[0])
        x = np.array([
            [-0.71881904, -0.23960676, -1.20534317],
            [-1.21582478, -0.88046404, -0.30952331]
        ], dtype=np.double)
        y = at.atCopyVect(x)
        assert_array_equal(x, y)

    def test_atInvVect(self):
        """\
        test for atInvVect
        """
        x = np.array([ 0.79539486,  0.43061085, -0.65511234], dtype=np.double)
        y = at.atInvVect(x)
        self.assertNotEqual(x[0], y[0])
        z = at.atInvVect(y)
        assert_array_almost_equal(x, z)

    def test_atAddVect(self):
        """\
        test for atAddVect / atSubVect
        """
        x = np.array([1.4609518 , 0.66283621, 0.87311351], dtype=np.double)
        y = np.array([-2.24894694,  1.83466292,  1.70731049], dtype=np.double)
        z = at.atAddVect(x, y)
        assert_array_almost_equal(z, x + y)
        x2 = at.atSubVect(z, y)
        assert_array_almost_equal(x, x2)

    def test_atMulVect(self):
        """\
        test for atMulVect / atDivVect
        """
        f = -1.39432165
        x = np.array([-0.75111902,  0.09063125, -1.73281983], dtype=np.double)
        y = at.atMulVect(f, x)
        assert_array_almost_equal(y, x * f)
        x2 = at.atDivVect(f, y)
        assert_array_almost_equal(x, x2)
        f = np.array([0.29481876, 1.18661088], dtype=np.double)
        x = np.array([
            [ 0.4818297 ,  0.76466069,  0.1724655 ],
            [ 0.27358087,  2.18051858, -0.03031352]
        ], dtype=np.double)
        y = at.atMulVect(f, x)
        x2 = at.atDivVect(f, y)
        assert_array_almost_equal(x, x2)

    def test_atMulAddVect(self):
        """\
        test for atMulAddVect
        """
        f = 0.39887174
        x = np.array([0.05243417, 1.78052024, 0.8211622 ], dtype=np.double)
        g = 0.05861536
        y = np.array([2.00929508, 0.60714111, 0.18076026], dtype=np.double)
        z = at.atMulAddVect(f, x, g, y)
        assert_array_almost_equal(z, [0.13869006, 0.745787  , 0.33813372])
        f = np.array([-1.04513677, -1.17868363], dtype=np.double)
        x = np.array([[0.76173523, 1.07852919, 0.70023528],
            [0.84146713, 1.50676144, 0.32285232]], dtype=np.double)
        g = np.array([0.68523773, 1.44409108], dtype=np.double)
        y = np.array([[ 1.81195902, -0.70817921,  1.23323953],
            [ 0.20359954,  0.85734256,  0.63763087]], dtype=np.double)
        z = at.atMulAddVect(f, x, g, y)
        assert_array_almost_equal(z, [
            [ 0.44550519, -1.61248163,  0.11322062],
            [-0.69780725, -0.5379143 ,  0.54025631]
        ])

    def test_atAngDistance(self):
        """\
        test for atAngDistance
        """
        x = np.array([0.09477122, 1.29953618, 0.27408418], dtype=np.double)
        y = np.array([-0.6891579 , -0.60434425, -1.63255041], dtype=np.double)
        r = at.atAngDistance(x, y)
        assert_almost_equal(r, 2.11849847)
        x = np.array([[-1.53417597, -0.48764878,  0.19689442],
            [ 0.56179407, -0.52919497, -0.23472045]], dtype=np.double)
        y = np.array([-0.6891579 , -0.60434425, -1.63255041], dtype=np.double)
        r = at.atAngDistance(x, y)
        assert_array_almost_equal(r, [1.22452379, 1.36012104])
        x = np.array([[ 1.78453164,  1.04969597, -1.05592802],
            [ 1.51926724,  1.09919702, -0.309648  ]], dtype=np.double)
        y = np.array([[-1.25362494,  0.23474269, -0.3029591 ],
            [ 0.99589005,  2.09646818, -0.62950257]], dtype=np.double)
        r = at.atAngDistance(x, y)
        assert_array_almost_equal(r, [2.15126017, 0.49942319])

    def test_atCrossPts(self):
        """\
        test for atCrossPts
        """
        x = np.array([ 2.99017204, -8.559615  ,  0.10421926], dtype=np.double)
        r1 = 13.69630381
        y = np.array([1.41822292, 7.32730667, 0.85217555], dtype=np.double)
        r2 = 3.84337717
        z = at.atCrossPts(x, r1, y, r2)
        assert_array_almost_equal(z, [
            [6.95311762e-310, 2.24735355e-314, 2.36124416e-314],
            [2.25033511e-314, 2.36124412e-314, 2.25033518e-314]
        ])
        x = np.array([
            [-5.02147513,  7.69688001,  1.55521638],
            [ 4.77490286,  1.9624724 , -3.20829281]
        ], dtype=np.double)
        r1 = np.array([1.45309074, 0.29263664], dtype=np.double)
        y = np.array([
            [-7.71387728,  4.61146647,  4.49669635],
            [ 3.83382434,  8.08519691,  3.64572596]
        ], dtype=np.double)
        r2 =np .array([-2.16299828, -5.10402487], dtype=np.double)
        z = at.atCrossPts(x, r1, y, r2)
        assert_array_almost_equal(z, [
            [[6.95312165e-310, 2.20699909e-314, 2.25404562e-314],
             [2.25404534e-314, 2.25404558e-314, 2.25404570e-314]],
            [[6.95312165e-310, 2.20699909e-314, 2.25404562e-314],
             [2.25404534e-314, 2.25404558e-314, 2.25404570e-314]]
        ])

    def test_atVectProd(self):
        """\
        test for atVectProd
        """
        x = np.array([-0.08798045,  0.06723859,  0.17361453], dtype=np.double)
        y = np.array([-0.60297527,  0.43874151, -0.20030532], dtype=np.double)
        z = at.atVectProd(x, y)
        assert_array_almost_equal(z, [-0.08964015, -0.12230822,  0.00194253])
        x = np.array([
            [ 2.1515796 ,  0.9837139 , -0.52759557],
            [-1.14507401,  1.47324569,  1.33680624]
        ], dtype=np.double)
        y = np.array([
            [-1.41833125, -0.71366302, -0.96825149],
            [ 1.0802284 , -1.51051502,  0.29825408]
        ], dtype=np.double)
        z = at.atVectProd(x, y)
        assert_array_almost_equal(z, [
            [-1.3290079 ,  2.83157544, -0.14027063],
            [ 2.45866744,  1.78557906,  0.13820966]
        ])

    def test_atScalProd(self):
        """\
        test for atScalProd
        """
        x = np.array([-0.79869465,  0.91728042,  1.15804255], dtype=np.double)
        y = np.array([-1.07578792, -0.16462245,  1.05789162], dtype=np.double)
        z = at.atScalProd(x, y)
        assert_almost_equal(z, 1.9333046154096298)
        x = np.array([[ 0.5853746 , -0.76206383, -0.08952434],
            [-1.68534189,  0.1208647 , -0.51673161]], dtype=np.double)
        y = np.array([[-0.25888345,  1.05238429,  1.04001202],
            [-1.15317707,  0.57720731,  0.74321669]], dtype=np.double)
        z = at.atScalProd(x, y)
        assert_array_almost_equal(z, [-1.04663419,  1.62921805])

    def test_atSetEuler(self):
        """\
        test for atSetEuler
        """
        z = np.array((-0.40248114,  1.0174042 , -1.44627077), dtype=at.AtPolarVect)
        y = np.array(( 0.50684814,  0.10141377, -0.51499413), dtype=at.AtPolarVect)
        x = at.atSetEuler(z, y)
        x2 = np.array((1.0174042 , 3.0170671 , 2.54904685), dtype=at.AtEulerAng)
        for name in ('phi', 'theta', 'psi'):
            assert_almost_equal(x[name], x2[name])

    def test_atRotPVect(self):
        """\
        test for atRotPVect
        """
        ea = np.array((-0.68173836,  0.16097662,  0.28885426), dtype=at.AtEulerAng)
        x = np.array((-0.21589573, -0.34986075, -0.08022128), dtype=at.AtPolarVect)
        y = at.atRotPVect(ea, x)
        assert_array_almost_equal([y['r'], y['lon'], y['lat']], [ 0.21589573,  3.18440031, -0.0720117 ])
        ea = np.array(
            [
                (-0.75039587,  0.73246903, -0.57343102),
                (-0.07335749,  1.64295957,  0.34290897)
            ],
             dtype=at.AtEulerAng
        )
        x = np.array(
            [
                (-0.0378821 ,  0.20482549, -0.7307556 ),
                (-1.27017179, -0.49023334,  0.95007705)
            ],
             dtype=at.AtPolarVect
        )
        y = at.atRotPVect(ea, x)
        assert_array_almost_equal(
            [
                [y[0]['r'], y[0]['lon'], y[0]['lat']],
                [y[1]['r'], y[1]['lon'], y[1]['lat']],
            ], [
                [ 0.0378821 ,  4.38589784,  0.21025908],
                [ 1.27017179,  6.21065204, -0.49130213]
            ]
        )

    def test_atVectToPol(self):
        """\
        test for atVectToPol / atPoToVect
        """
        x = np.array([-0.77492264,  0.62741098,  1.70959476], dtype=np.double)
        y = at.atVectToPol(x)
        assert_array_almost_equal(
            [y['r'], y['lon'], y['lat']],
            [1.97910684, 2.4609992 , 1.04280615]
        )
        x2 = at.atPolToVect(y)
        assert_array_almost_equal(x, x2)
        x = np.array([[-2.3622941 , -0.42425342, -0.90111249],
            [ 2.0719562 ,  0.09250148, -1.42842767]], dtype=np.double)
        y = at.atVectToPol(x)
        x2 = at.atPolToVect(y)
        assert_array_almost_equal(x, x2)

    def test_atConvPol(self):
        x = np.array(((18, 36, 56.33635), (1, 38, 47, 1.2802)), dtype=at.AtPolarVect60)
        y = at.atConvPol(x)
        assert_array_almost_equal(
            [y['lon'], y['lat']],
            [4.873565508047745, 0.6769030681498797],
            9
        )
        x = np.array(
            [
                ((18, 36, 56.33635), (1, 38, 47, 1.2802)),
                ((5, 34, 31.94), (1, 22, 0, 52.2)),
            ],
            dtype=at.AtPolarVect60
        )
        y = at.atConvPol(x)
        assert_array_almost_equal(
            [
                [y[0]['lon'], y[0]['lat']],
                [y[1]['lon'], y[1]['lat']],
            ],
            [
                (4.87356551, 0.67690307),
                (1.45967267, 0.38422551),
            ],
            8
        )

    def test_atTimeToAtTimeD(self):
        """\
        test for atTimeToAtTimeD
        """
        x = np.array(
            (2020, 1, 5, 12, 18, 20, 300),
            dtype=at.AtTime)
        y = at.atAtTimeToAtTimeD(x)
        y2 = np.array(
            (2020, 1, 5, 12, 18, 20, 0.3),
            dtype=at.AtTimeD)
        x = np.array(
            [
                (2020, 1, 5, 12, 18, 20, 300),
                (2020, 2, 8, 20, 15, 11, 200),
            ],
            dtype=at.AtTime)
        y = at.atAtTimeToAtTimeD(x)
        y2 = np.array(
            [
                (2020, 1, 5, 12, 18, 20, 0.3),
                (2020, 2, 8, 20, 15, 11, 0.2),
            ],
            dtype=at.AtTimeD
        )
        self.assertEqual(y[0], y2[0])
        self.assertEqual(y[1], y2[1])
        x2 = at.atAtTimeDToAtTime(y)
        self.assertEqual(x[0], x2[0])
        self.assertEqual(x[1], x2[1])

    def test_atReformatAtTime(self):
        """\
        Test for atReformatAtTime
        """
        time = np.array((2020, 1, 4, 25, 18, 20, 300), dtype=at.AtTime)
        time2 = at.atReformatAtTime(time)
        assert_array_equal(
            [time2['yr'], time2['mo'], time2['dy'], time2['hr'], time2['mn'], time2['sc'], time2['ms']],
            [2020, 1, 5, 1, 18, 20, 300],
        )

    def test_atReformatAtTimeD(self):
        """\
        test for atReformatAtTimeD
        """
        time = np.array((2020, 1, 4, 25, 18, 20, 0.3), dtype=at.AtTimeD)
        time2 = at.atReformatAtTimeD(time)
        assert_array_equal(
            [time2['yr'], time2['mo'], time2['dy'], time2['hr'], time2['mn'], time2['sc'], time2['ss']],
            [2020, 1, 5, 1, 18, 20, 0.3]
        )

    def test_atCTime(self):
        """\
        test for atCTime
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 300), dtype=at.AtTime)
        ctime = at.atCTime(x)
        self.assertEqual(ctime, '20/01/05 12:18:20.300000')

    def test_atCTimeD(self):
        """\
        test for atCTimeD
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 0.3), dtype=at.AtTimeD)
        ctime = at.atCTimeD(x)
        self.assertEqual(ctime, '20/01/05 12:18:20.300000')

    def test_atCTime2(self):
        """\
        test for atCTime2
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 300), dtype=at.AtTime)
        ctime = at.atCTime2(x)
        self.assertEqual(ctime, '2020/01/05 12:18:20.300000')

    def test_atCTimeD2(self):
        """\
        test for atCTimeD2
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 0.3), dtype=at.AtTimeD)
        ctime = at.atCTimeD2(x)
        self.assertEqual(ctime, '2020/01/05 12:18:20.300000')

    def test_atMJulian(self):
        """\
        test for atMJulian
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 300), dtype=at.AtTime)
        mjd = at.atMJulian(x)
        self.assertEqual(mjd, 58853.51273495371)

    def test_atMJulianD(self):
        """\
        test for atMJulianD
        """
        x = np.array((2020, 1, 5, 12, 18, 20, 0.3), dtype=at.AtTimeD)
        mjd = at.atMJulianD(x)
        self.assertEqual(mjd, 58853.51273495371)

    def test_atMJDate(self):
        """\
        test for atMJDate
        """
        mjd = np.array([58853.239847, 59374.9034745], dtype=np.double)
        time = at.atMJDate(mjd)
        assert_array_almost_equal(
            [
                [time[0]['yr'], time[0]['mo'], time[0]['dy'], time[0]['hr'], time[0]['mn'], time[0]['sc'], time[0]['ms']],
                [time[1]['yr'], time[1]['mo'], time[1]['dy'], time[1]['hr'], time[1]['mn'], time[1]['sc'], time[1]['ms']],
            ],
            [[2020, 1, 5,  5, 45, 22, 780.79974], [2021, 6, 9, 21, 41,  0, 196.80023]],
            5
        )

    def test_atMJDateD(self):
        """\
        test for atMJDate
        """
        mjd = np.array([58853.239847, 59374.9034745], dtype=np.double)
        time = at.atMJDateD(mjd)
        assert_array_almost_equal(
            [
                [time[0]['yr'], time[0]['mo'], time[0]['dy'], time[0]['hr'], time[0]['mn'], time[0]['sc'], time[0]['ss']],
                [time[1]['yr'], time[1]['mo'], time[1]['dy'], time[1]['hr'], time[1]['mn'], time[1]['sc'], time[1]['ss']],
            ],
            [[2020, 1, 5,  5, 45, 22, 0.78079974], [2021, 6, 9, 21, 41,  0, 0.19680023]],
            5
        )

    def test_atMissionTimeInit(self):
        """\
        test for atMissionTimeInit/atMissionTimeResetTable
        """
        at.atMissionTimeInit(os.path.join(os.environ['HEADAS'], 'refdata', 'leapsec.fits'), 0)
        at.atMissionTimeResetTable()

    def test_atAtTimeDToMission(self):
        """\
        test for atTimeDToMission/atMissionToAtTimeD
        """
        at.atMissionTimeInit(os.path.join(os.environ['HEADAS'], 'refdata', 'leapsec.fits'), 0)
        mjd = 58853.239847
        time = at.atMJDateD(mjd)
        mission = at.atAtTimeDToMission(at.optional.MAXI_MJD_BASE, time)
        assert_almost_equal(mission * 1.0e-8, 6.31518328)
        mjd = np.array([58853.239847, 59374.9034745], dtype=np.double)
        time = at.atMJDateD(mjd)
        mission = at.atAtTimeDToMission(at.optional.MAXI_MJD_BASE, time)
        assert_array_almost_equal(mission * 1.0e-8, [6.31518328, 6.76590065])
        at.atMissionTimeResetTable()

    def test_atMJDToMission(self):
        """\
        test for atMJDToMission/atMissionToMJD
        """
        at.atMissionTimeInit(os.path.join(os.environ['HEADAS'], 'refdata', 'leapsec.fits'), 0)
        mjd = 59000
        maxi = at.atMJDToMission(at.optional.MAXI_MJD_BASE, mjd)
        mjd2 = at.atMissionToMJD(at.optional.MAXI_MJD_BASE, maxi)
        assert_array_equal(mjd, mjd2)

    def test_atPrecession(self):
        """\
        test for atPrecession
        """
        mjd0 = 0.19531328
        x0 = np.array([ 0.03251248, -0.04954698,  0.00132439], dtype=np.double)
        mjd = -0.75180727
        x = at.atPrecession(mjd0, x0, mjd)
        assert_array_almost_equal(x, [ 0.03251245, -0.049547  ,  0.00132438])
        mjd0 = np.array([0.40987937, 1.17953872], dtype=np.double)
        x0 = np.array([
            [-1.60538473,  0.74059943,  0.76352152],
            [ 1.5260757 , -2.14613407, -1.07234361]
        ], dtype=np.double)
        mjd = np.array([0.09509402, 0.42226205], dtype=np.double)
        x = at.atPrecession(mjd0, x0, mjd)
        assert_array_almost_equal(x, [
            [-1.60538452,  0.74059974,  0.76352165],
            [ 1.52607449, -2.14613478, -1.07234392]
        ])

    def test_atPrecessRM(self):
        """\
        test for atPrecessRM
        """
        mjd0 = 50000
        mjd = 60000
        x = at.atPrecessRM(mjd0, mjd)
        assert_array_almost_equal(x, [
            [ 9.99977718e-01, -6.12261101e-03, -2.66029018e-03],
            [ 6.12261101e-03,  9.99981257e-01, -8.14443496e-06],
            [ 2.66029018e-03, -8.14366847e-06,  9.99996461e-01]])
        mjd0 = np.array([50000, 60000], dtype=np.double)
        mjd = np.array([60000, 90000], dtype=np.double)
        x = at.atPrecessRM(mjd0, mjd)
        assert_array_almost_equal(x, [
            [[ 9.99977718e-01, -6.12261101e-03, -2.66029018e-03],
             [ 6.12261101e-03,  9.99981257e-01, -8.14443496e-06],
             [ 2.66029018e-03, -8.14366847e-06,  9.99996461e-01]],
            [[ 9.99799371e-01, -1.83728305e-02, -7.97853524e-03],
             [ 1.83728304e-02,  9.99831203e-01, -7.33118392e-05],
             [ 7.97853543e-03, -7.32911440e-05,  9.99968168e-01]]])

    def test_atPrecessEuler(self):
        """\
        test for atPrecessEuler
        """
        mjd0 = 50000
        mjd = 60000
        ea = at.atPrecessEuler(mjd0, mjd)
        assert_array_almost_equal(
            [ea['phi'], ea['theta'], ea['psi']],
            [6.28012412e+00, 2.66030579e-03, 6.28012383e+00]
        )
        mjd0 = np.array([50000, 60000], dtype=np.double)
        mjd = np.array([60000, 90000], dtype=np.double)
        ea = at.atPrecessEuler(mjd0, mjd)
        assert_array_almost_equal(
            [
                [ea[0]['phi'], ea[0]['theta'], ea[0]['psi']],
                [ea[1]['phi'], ea[1]['theta'], ea[1]['psi']],
            ],
            [
                [6.28012412e+00, 2.66030579e-03, 6.28012383e+00],
                [6.27399953e+00, 7.97895671e-03, 6.27399693e+00]
            ]
        )

    def test_atPrecessRMJ2000(self):
        """\
        test for atPrecessRMJ2000
        """
        mjd = 33281.923
        ea = at.atPrecessRMJ2000(mjd)
        assert_array_almost_equal(
            [ea['phi'], ea['theta'], ea['psi']],
            [6.27759617e+00, 4.85909898e-03, 6.27759521e+00]
        )
        mjd = np.array([33281.923, 95000.0], dtype=np.double)
        ea = at.atPrecessRMJ2000(mjd)
        assert_array_almost_equal(
            [
                [ea[0]['phi'], ea[0]['theta'], ea[0]['psi']],
                [ea[1]['phi'], ea[1]['theta'], ea[1]['psi']],
            ],
            [
                [6.27759617e+00, 4.85909898e-03, 6.27759521e+00],
                [3.15490271e+00, 1.15577061e-02, 3.15489727e+00]
            ]
        )

    def test_atSaisa(self):
        """\
        test for atSaisa
        """
        mjd0 = 58000.0
        pv0 = np.array((-0.40575718, -0.93873994, -1.33984516), dtype=at.AtPolarVect)
        mjd = 58180.0
        pv = at.atSaisa(mjd0, pv0, mjd)
        assert_array_almost_equal(
            [pv['lon'], pv['lat']],
            [5.34471979, -1.33981687])
        mjd0 = np.array([32000.0, 35000.0], dtype=np.double)
        pv0 = np.array(
            [
                (-1.12261212, -1.16313735, -1.09833854),
                (-0.76886492,  0.30742754,  0.01168822)
            ], dtype=at.AtPolarVect
        )
        mjd = np.array([58000.0, 98000.0], dtype=np.double)
        pv = at.atSaisa(mjd0, pv0, mjd)
        assert_array_almost_equal(
            [
                [pv[0]['lon'], pv[0]['lat']],
                [pv[1]['lon'], pv[1]['lat']],
            ],
            [
                [5.14826455, -1.09550725],
                [0.34610529,  0.0275548],
            ])

    def test_atGeodcr(self):
        """\
        test for atGeodcr
        """
        mjd = 34000.0
        x = np.array([-0.46457987,  0.96301326,  0.14642522], dtype=np.double)
        y = at.atGeodcr(mjd, x)
        assert_array_almost_equal(y, [-6.35659259e+03,  4.88399785e-01,  1.54592108e+00], 5)
        mjd = np.array([34000.0, 58000.0], dtype=np.double)
        x = np.array([[-2.05745387,  1.60547813, -0.83360796],
            [-0.45148306, -0.126567  , -1.7481337 ]])
        y = at.atGeodcr(mjd, x)
        assert_array_almost_equal(
            [
                [y[0][0], y[1][0], y[2][0]],
                [y[0][1], y[1][1], y[2][1]],
            ], [
                [-6.35584074e+03,  9.47081667e-01, -1.51100859e+00],
                [-6.35500172e+03,  3.70625725e+00, -1.56028047e+00]
            ],
            5
        )

    def test_atGeodetic(self):
        """\
        test for atGeodetic
        """
        mjd = 34000.0
        x = np.array([-0.46457987,  0.96301326,  0.14642522], dtype=np.double)
        y = at.atGeodetic(mjd, x)
        assert_array_almost_equal(y, [0.94421097, 0.50169178, 0.14642522])
        mjd = np.array([34000.0, 58000.0], dtype=np.double)
        x = np.array([[-2.05745387,  1.60547813, -0.83360796],
            [-0.45148306, -0.126567  , -1.7481337 ]])
        y = at.atGeodetic(mjd, x)
        assert_array_almost_equal(y, [[ 1.52422277,  2.11835347, -0.83360796],
            [-0.39610182, -0.25091733, -1.7481337 ]])

    def test_atInvGeodetic(self):
        """\
        test for atInvGeodetic
        """
        mjd = 34000.0
        x = np.array([-0.46457987,  0.96301326,  0.14642522], dtype=np.double)
        y = at.atGeodetic(mjd, x)
        assert_array_almost_equal(y, [0.94421097, 0.50169178, 0.14642522])
        mjd = np.array([34000.0, 58000.0], dtype=np.double)
        x = np.array([[-2.05745387,  1.60547813, -0.83360796],
            [-0.45148306, -0.126567  , -1.7481337 ]])
        y = at.atGeodetic(mjd, x)
        assert_array_almost_equal(y, [
            [ 1.52422277,  2.11835347, -0.83360796],
            [-0.39610182, -0.25091733, -1.7481337 ]])

    def test_atEllipsoid(self):
        """\
        test for atEllipsoid
        """
        xp = np.array(( 3.50846977,  0.26318369, -0.61350897), dtype=at.AtPolarVect)
        pos = at.atEllipsoid(xp)
        assert_array_almost_equal(pos, [-1.50681305e+00, -6.35464063e+03], 5)

    def test_atSetGeoRM(self):
        """\
        test for atSetGeoRM
        """
        mjd = 34000.0
        rm = at.atSetGeoRM(mjd)
        assert_array_almost_equal(rm, [
            [ 0.03890247,  0.99924301,  0.        ],
            [-0.99924301,  0.03890247,  0.        ],
            [ 0.        ,  0.        ,  1.        ]])
        mjd = np.array([34000.0, 58000.0], dtype=np.double)
        rm = at.atSetGeoRM(mjd)
        assert_array_almost_equal(rm, [
            [[ 0.03890247,  0.99924301,  0.        ],
             [-0.99924301,  0.03890247,  0.        ],
             [ 0.        ,  0.        ,  1.        ]],
            [[ 0.95785862, -0.28724009,  0.        ],
             [ 0.28724009,  0.95785862,  0.        ],
             [ 0.        ,  0.        ,  1.        ]]])

    def test_atSidereal(self):
        """\
        test for atSidereal
        """
        mjd = 34000.0
        rm = at.atSidereal(mjd)
        assert_almost_equal(rm, 1.5318840378912069)
        mjd = np.array([34000.0, 60000.0], dtype=np.double)
        rm = at.atSidereal(mjd)
        assert_array_almost_equal(rm, [1.53188404, 2.69831297])

    def test_atKepler(self):
        """\
        test for atKepler
        """
        g = 0.61899078
        eccent = 0.19171448
        e = at.atKepler(g, eccent)
        assert_almost_equal(e, 0.7496170760554403)
        g = np.array([-0.72810668,  0.76750196], dtype=np.double)
        eccent = np.array([-0.17945859,  0.45408998], dtype=np.double)
        e = at.atKepler(g, eccent)
        assert_array_almost_equal(e, [-0.6233467 ,  1.18887485])

    def test_atOrbPlane(self):
        """\
        test for atOrbPlane
        """
        x = np.array([-1.45654522, -0.27725439,  0.17012062], dtype=np.double)
        ol = -2.19181983
        ob = -0.47275093
        ai = 0.06300339
        y = at.atOrbPlane(x, ol, ob, ai)
        assert_array_almost_equal(y, [1.16055514, 0.90314139, 0.25452386])
        x = np.array([[-0.92411905,  1.32637964,  0.42807901],
            [-0.60010767,  1.26583505,  0.89701666]], dtype=np.double)
        ol = np.array([0.05029834, 1.34522057], dtype=np.double)
        ob = np.array([-1.51607726, -0.99137318], dtype=np.double)
        ai = np.array([0.42531346, 1.11707383], dtype=np.double)
        y = at.atOrbPlane(x, ol, ob, ai)
        assert_array_almost_equal(y, [
            [ 0.93212954,  1.04217654,  0.91735118],
            [-1.53437454,  0.63082023,  0.12193073]])

    def test_atGeographic(self):
        """\
        test for atGeographic
        """
        y = np.array(( 1.04106063, -0.80847329, -0.3965017 ), dtype=at.AtPolarVect)
        z = at.atGeographic(y)
        assert_array_almost_equal(z, [ 4065.72433361, -4257.82539404, -2448.33191374], 6)
        y = np.array(
            [
                (-1.43964913,  0.03938443, -0.06693507),
                ( 0.10593706,  0.27399077,  1.83075425),
            ],
            dtype=at.AtPolarVect
        )
        z = at.atGeographic(y)
        assert_array_almost_equal(z, [
            [ 6357.57929283,   250.51917957,  -423.65654481],
            [-1583.26589591,  -444.9916877 ,  6141.90561969]], 6)

    def test_atGeographicToGeodetic(self):
        """\
        test for atGeographicToGeodetic
        """
        y = np.array((-6.35512329e+03,  5.25751834e+00,  1.56732620e+00), dtype=at.AtPolarVect)
        z = at.atGeographicToGeodetic(y)
        assert_array_almost_equal(z, [ 0.08001753, -0.13194989,  1.62875639], 5)
        y = np.array([
            (-6.35458362e+03,  4.03635623e+00,  1.52716737e+00),
            (-6.35666743e+03,  2.61441873e+00, -1.55292813e+00)], dtype=at.AtPolarVect)
        z = at.atGeographicToGeodetic(y)
        assert_array_almost_equal(z, [
            [-1.22720899, -1.52995541,  2.12592154],
            [-0.66273645,  0.38579679, -0.0780334 ]], 5)

    def test_atGeodesic(self):
        """\
        test for atGeodesic
        """
        y = np.array((-0.94582466,  0.53661068, -0.86579929), dtype=at.AtPolarVect)
        z = at.atGeodesic(y)
        assert_array_almost_equal(z, [ 3558.68323969,  2116.81792261, -4833.83126562])
        y = np.array(
            [
                ( 0.00861286,  0.36167557, -0.21067305),
                ( 0.91139409, -0.25040311,  2.45428233),
            ],
            dtype=at.AtPolarVect
        )
        z = at.atGeodesic(y)
        assert_array_almost_equal(z, [
            [ 5834.47198115,  2207.28007619, -1325.05090586],
            [-4783.39148913,  1223.4545371 ,  4025.59099321]])

    def test_atGeodeticToGeographic(self):
        """\
        test for atGeodeticToGeographic
        """
        vect = np.array([ 0.08001731, -0.13194953,  1.62875845], dtype=np.double)
        pv = at.atGeodeticToGeographic(vect)
        assert_array_almost_equal(
            [pv['r'], pv['lon'], pv['lat']],
            [-6.35512329e+03,  5.25751834e+00,  1.56732620e+00],
            5
        )
        vect = np.array([
            [-1.22720872, -1.52995508,  2.12591695],
            [-0.66273628,  0.38579669, -0.07803197]], dtype=np.double)
        pv = at.atGeodeticToGeographic(vect)
        assert_array_almost_equal(
            [
                [pv[0]['r'], pv[0]['lon'], pv[0]['lat']],
                [pv[1]['r'], pv[1]['lon'], pv[1]['lat']],
            ],
            [
                [-6.35458362e+03,  4.03635623e+00,  1.52716737e+00],
                [-6.35666743e+03,  2.61441873e+00, -1.55292813e+00],
            ],
            5
        )

    def test_atGroundCoord(self):
        """\
        test for atGroundCoord
        """
        station = np.array([ 0.23425293, -0.08897633,  0.16462214], dtype=np.double)
        rm = at.atGroundCoord(station)
        assert_array_almost_equal(rm, [
            [[-0.51329151,  0.1949636 ,  0.8357757 ],
             [-0.35507903, -0.93483629,  0.        ],
             [ 0.78131346, -0.29676642,  0.54907101]]])
        station = np.array([
            [ 0.53192123,  0.77845497,  1.14852435],
            [-1.4406765 ,  0.78669302,  1.68215248]], dtype=np.double)
        rm = at.atGroundCoord(station)
        assert_array_almost_equal(rm, [
            [[-0.43606342, -0.63816919,  0.63449884],
             [ 0.825656  , -0.56417388,  0.        ],
             [ 0.35796767,  0.52387778,  0.77292381]],
            [[ 0.62815721, -0.34301031,  0.69839992],
             [ 0.47926031,  0.87767281, -0.        ],
             [-0.61296661,  0.33471536,  0.71570773]]])

    def test_atAzElSet(self):
        """\
        test for atAzElSet
        """
        y = np.array((6500.0, -0.74143729, -0.30165612), dtype=at.AtPolarVect)
        vStation, stationRM = at.atAzElSet(y)
        assert_array_almost_equal(vStation, [ 9070.07897661, -8305.73075869, -3813.99045889])
        assert_array_almost_equal(stationRM, [
            [ 0.21911229, -0.20064739,  0.95484576],
            [-0.67534861, -0.73749865,  0.        ],
            [ 0.70419746, -0.64485376, -0.29710195]])
        y = np.array(
            [
                (6500.0,  0.75065593,  0.0635082 ),
                (7000.0,  0.21583328, -1.49211652),
            ],
            dtype=at.AtPolarVect
        )
        vStation, stationRM = at.atAzElSet(y)
        assert_array_almost_equal(vStation, [
            [  9398.10788274,   8766.76557417,    814.61321931],
            [  1028.74397025,    225.55045863, -13315.29915174]])
        assert_array_almost_equal(stationRM, [
            [[-4.64086268e-02, -4.32910067e-02,  9.97984032e-01],
             [ 6.82118550e-01, -7.31241604e-01,  2.08166817e-17],
             [ 7.29767445e-01,  6.80743421e-01,  6.34655174e-02]],
            [[ 9.73776396e-01,  2.13498906e-01,  7.85986539e-02],
             [ 2.14161449e-01, -9.76798277e-01, -3.46944695e-18],
             [ 7.67750297e-02,  1.68328016e-02, -9.96906340e-01]]])

    def test_atAzEl(self):
        """\
        test for atAzEl
        """
        x = np.array([-0.39397016, -0.39464641,  0.69937677], dtype=np.double)
        stationV = np.array([-0.98526279, -0.65743303,  1.03370392], dtype=np.double)
        stationRM = np.array([[ 2.12702668, -0.08692043,  0.08660912],
            [ 1.16994218, -0.44640247,  0.49252039],
            [-0.18998812,  1.44178854,  0.227753  ]], dtype=np.double)
        y = at.atAzEl(x, stationV, stationRM)
        assert_array_almost_equal(
            [y['r'], y['lon'], y['lat']],
            [1.28778237, 5.95559446, 0.14839522])
        x = np.array([[ 0.37621902,  1.84986529, -0.53761347],
               [-1.39144555, -0.41275141, -0.02539562]], dtype=np.double)
        stationV = np.array([[ 1.98331304, -1.49496601,  0.60563758],
               [ 1.7111671 , -0.65969101,  0.31776543]], dtype=np.double)
        stationRM = np.array([[[ 0.58910497, -0.28170759, -0.1323472 ],
                [ 0.45918325, -0.20477654, -0.4248073 ],
                [ 1.4011292 ,  0.95325439, -0.1523251 ]],
               [[ 1.3163081 , -0.2100887 ,  0.56431073],
                [-1.23947967,  0.71258608, -1.29402421],
                [-0.6100833 ,  0.71664867,  0.09278621]]], dtype=np.double)
        y = at.atAzEl(x, stationV, stationRM)
        assert_array_almost_equal(
            [
                [y[0]['r'], y[0]['lon'], y[0]['lat']],
                [y[1]['r'], y[1]['lon'], y[1]['lat']],
            ],
            [
                [2.26540649, 2.6469624 , 0.51250777],
                [6.5452395 , 3.94246704, 0.31663263]
            ]
        )

    def test_atEarthOccult(self):
        """\
        test for atEarthOccult
        TODO: correct input/output value
        """
        satVect = np.array([0.02999667, 0.39413344, 0.06361616])
        xVect = np.array([0.50000144, 0.04252517, 0.59291179])
        sunVect = np.array([0.67528635, 0.70146891, 0.01250719])
        flag, el = at.atEarthOccult(satVect, xVect, sunVect)
        satVect = np.array([[0.10826278, 0.40471944, 0.11134734],
            [0.59264702, 0.226126  , 0.45085271]])
        xVect = np.array([[0.46530041, 0.15406309, 0.88482197],
            [0.62858961, 0.74743529, 0.11572397]])
        sunVect = np.array([[0.30375175, 0.63905823, 0.11424469],
            [0.42574007, 0.36987773, 0.20262485]])
        flag, el = at.atEarthOccult(satVect, xVect, sunVect)

    def test_atRigSet2(self):
        """\
        test for AtRigSet2
        """
        if 'CALDB' in os.environ:
            rs = at.atRigSet2(os.path.join(os.environ['CALDB'], 'data/gen/bcf/rigidity_20160101.fits'))
            self.assertIsNotNone(rs)
            x = np.array((0.70775397, 0.09307583, 0.96839853), dtype=at.AtPolarVect)
            y = at.atRigidity2(rs, x)
            assert_almost_equal(y, 774187862.8771715)
            x = np.array([(0.74885502, 0.93495123, 0.63251439),
                          (0.98621336, 0.91344872, 0.46091802)],
                         dtype=at.AtPolarVect)
            y = at.atRigidity2(rs, x)
            assert_array_almost_equal(
                y*1.0e-10,
                np.array([6.79995180e+08, 5.43097683e+08])*1.0e-10)
            at.atRigFree2(rs)

    # #def atGeomagSet(double mjd, int nmax):
    # #def atGeomag(np.ndarray x, np.ndarray xs):

    def test_atSun(self):
        """\
        test for atSun
        """
        mjd = 58849.0
        pos = at.atSun(mjd)
        assert_array_almost_equal(pos, [ 0.16637608, -0.8892618 , -0.38565779])
        mjd = np.array([62502.0, 69807.0], dtype=np.double)
        pos = at.atSun(mjd)
        assert_array_almost_equal(pos, [
            [ 0.17387436, -0.88804621, -0.38513053],
            [ 0.17164496, -0.88841461, -0.38529032]])

    def test_atPlanet(self):
        """\
        test for atPlanet
        """
        mjd = 58849.0
        pos, size, mag = at.atPlanet(mjd)
        assert_array_almost_equal(pos, [
            [  0.1030447 ,  -1.29934128,  -0.59799788],
            [  0.88959144,  -0.82468755,  -0.40219786],
            [  0.16637025,  -0.88919401,  -0.38546624],
            [ -1.15382143,  -1.70223135,  -0.736609  ],
            [  0.69609522,  -5.66341794,  -2.44540564],
            [  3.95047597,  -9.42474827,  -4.07027835],
            [ 16.19869151,   9.85924289,   4.09503902],
            [ 29.24856068,  -6.72902951,  -3.49931176],
            [ 13.51484224, -29.60379657, -13.36553441]])
        assert_array_almost_equal(size, [
            1.13689733e-05, 3.16553967e-05, 4.73125518e-03,
            1.03954479e-05, 7.68796157e-05, 3.64614890e-05,
             8.75185012e-06, 5.37581958e-06, 2.85011805e-07])
        assert_array_almost_equal(mag, [
            0.28597761,  -4.16183584, -26.55      ,
            1.4032496 , -1.37416346,   1.53510198,
            6.07402617,   7.7241355 , 14.4030855 ])
        mjd = np.array([62502.0, 69807.0], dtype=np.double)
        pos, size, mag = at.atPlanet(mjd)
        assert_array_almost_equal(pos, [
            [[ 1.05556136e-01, -6.21891897e-01, -2.35690948e-01],
             [ 8.80510438e-02, -2.38774413e-01, -8.73556089e-02],
             [ 1.73870996e-01, -8.87991856e-01, -3.84921711e-01],
             [ 1.45304935e+00, -1.36476475e+00, -5.96130404e-01],
             [-3.84397042e+00, -4.26798421e+00, -1.73601436e+00],
             [ 5.69331446e+00,  5.89806533e+00,  2.17903512e+00],
             [ 4.40809949e+00,  1.63194965e+01,  7.09185263e+00],
             [ 2.95102168e+01,  3.92199437e+00,  8.54025158e-01],
             [ 2.35808270e+01, -2.51820759e+01, -1.50188602e+01]],

            [[-7.87962245e-03, -6.57930837e-01, -2.43334979e-01],
             [ 3.13447364e-01, -1.53570699e+00, -6.85340136e-01],
             [ 1.71648936e-01, -8.88383287e-01, -3.85044522e-01],
             [-1.37170427e+00, -1.35156211e+00, -5.83085889e-01],
             [-2.23311775e+00,  3.37820251e+00,  1.50302718e+00],
             [ 4.89324843e+00, -8.97773795e+00, -3.92591370e+00],
             [-1.77247460e+01,  2.55664929e+00,  1.37745461e+00],
             [ 1.75654531e+01,  2.16137353e+01,  8.39264644e+00],
             [ 3.79317583e+01, -1.12727578e+01, -1.50097612e+01]]])
        assert_array_almost_equal(size, [
            [2.42117170e-05, 1.50353172e-04, 4.73126468e-03, 1.09134223e-05,
             7.95382349e-05, 4.72838533e-05, 9.26201053e-06, 5.45415900e-06,
             2.66484073e-07],
            [2.32401526e-05, 2.36489243e-05, 4.73121056e-03, 1.12858628e-05,
             1.10490835e-04, 3.66196361e-05, 9.45313405e-06, 5.58420021e-06,
             2.36916589e-07]])
        assert_array_almost_equal(mag, [
            [ -2.2234493 ,  -7.5661421 , -26.55      ,   0.99261628,
             -1.36847234,   0.76141984,   5.88630966,   7.68748187,
             14.70510761],
            [ -2.05245097,  -3.52490959, -26.55      ,   1.27041438,
             -2.15246608,   1.52047487,   5.73510257,   7.63710727,
             15.23892183]])

    def test_atMoon(self):
        """\
        test for atMoon
        """
        mjd = 58849.0
        pos, size, phase, distan = at.atMoon(mjd)
        assert_array_almost_equal(pos, [390167.78541501, -76536.28522558, -70774.78779382])
        assert_almost_equal(size, 0.0043035125493806154)
        assert_almost_equal(phase, 1.1375550619185972)
        assert_almost_equal(distan, 403853.6545817566)
        mjd = np.array([62502.0, 69807.0], dtype=np.double)
        pos, size, phase, distan = at.atMoon(mjd)
        assert_array_almost_equal(pos, [
            [-193046.06731509, -277277.24576727, -136877.45149175],
            [ 359570.15487198,   98046.72274222,   66917.83409257]])
        assert_array_almost_equal(size, [0.0047677 , 0.00458986])
        assert_array_almost_equal(phase, [5.54985976, 1.59532833])
        assert_array_almost_equal(distan, [364533.80070026, 378657.96259261])

    # #def test_atSAA(self):

    def test_atBrazil(self):
        """\
        test for atBrazil
        """
        lon = 0.47308029
        lat = 0.57541424
        flag = at.atBrazil(lon, lat)
        self.assertEqual(flag, 0)
        lon = np.array([-0.5978917 ,  1.24801046], dtype=np.double)
        lat = np.array([-0.31617633,  1.25636903], dtype=np.double)
        flag = at.atBrazil(lon, lat)
        assert_array_equal(flag, [1, 0])

    def test_atSISBrazil(self):
        """\
        test for atSISBrazil
        """
        lon = 0.51143235
        lat = -0.72462475
        flag = at.atSISBrazil(lon, lat)
        self.assertEqual(flag, 1)
        lon = np.array([ 0.80418655, -0.08120426], dtype=np.double)
        lat = np.array([ 0.0883646, -0.3369461], dtype=np.double)
        flag = at.atSISBrazil(lon, lat)
        assert_array_equal(flag, [0, 1])

    def test_atSTTBrazil(self):
        """\
        test for atSTTBrazil
        """
        lon = 0.51143235
        lat = -0.72462475
        flag = at.atSTTBrazil(lon, lat)
        self.assertEqual(flag, 0)
        lon = np.array([ 0.80418655, -0.08120426], dtype=np.double)
        lat = np.array([ 0.0883646, -0.3369461], dtype=np.double)
        flag = at.atSTTBrazil(lon, lat)
        assert_array_equal(flag, [0, 0])

    def test_atHXDBrazil(self):
        """\
        test for atHXDBrazil
        """
        lon = 0.51143235
        lat = -0.72462475
        flag = at.atHXDBrazil(lon, lat)
        self.assertEqual(flag, 0)
        lon = np.array([ 0.80418655, -0.08120426], dtype=np.double)
        lat = np.array([ 0.0883646, -0.3369461], dtype=np.double)
        flag = at.atHXDBrazil(lon, lat)
        assert_array_equal(flag, [0, 1])

    # #def test_atEarthElev(self):

    def test_atAberration(self):
        """\
        test for atAberration / atInvAberration
        """
        mjd = 58849.0
        x0 = np.array([1.09965976, 1.28493194, 0.37601414], dtype=np.double)
        x = at.atAberration(mjd, x0)
        assert_array_almost_equal(x, [1.09970481, 1.28508359, 0.37605634])
        x2 = at.atInvAberration(mjd, x)
        assert_array_almost_equal(x0, x2)
        mjd = np.array([62502.0, 69807.0], dtype=np.double)
        x0 = np.array([
            [-0.57825165,  0.06892746, -0.98411623],
            [-0.53112574, -0.44465916,  0.11125614]], dtype=np.double)
        x = at.atAberration(mjd, x0)
        assert_array_almost_equal(x, [
            [-0.57831341,  0.06890705, -0.98406189],
            [-0.53119263, -0.44464916,  0.11124276]])
        x2 = at.atInvAberration(mjd, x)
        assert_array_almost_equal(x0, x2)

    def test_atJ2000toB1950(self):
        """
        test for atJ2000toB1950 and reverse
        """
        pos = np.array([ 1.56262169, -0.99250159], dtype=np.double)
        pos2 = at.atJ2000toB1950(pos)
        assert_array_almost_equal(pos2, [ 0.92205524, -1.270852  ])
        pos3 = at.atB1950toJ2000(pos2)
        assert_array_almost_equal(pos3, pos)
        pos = np.array([[0.61113454, 0.7413489 ],
            [1.09887371, 0.51650754]], dtype=np.double)
        pos2 = at.atJ2000toB1950(pos)
        assert_array_almost_equal(pos2, [
            [3.59970432e+02, 4.62939445e-01],
            [4.58162344e-01, 2.38119840e-01]])
        pos3 = at.atB1950toJ2000(pos2)
        assert_array_almost_equal(pos3, pos)

    def test_atJ2000toGal(self):
        """\
        test for atJ2000toGal and reverse
        """
        pos = np.array([ 6.35801955, 41.75090981], dtype=np.double)
        pos2 = at.atJ2000toGal(pos)
        assert_array_almost_equal(pos2, [117.745388, -20.858796])
        pos3 = at.atGaltoJ2000(pos2)
        assert_array_almost_equal(pos3, pos)
        pos = np.array([[  8.93388231, -29.38051172],
            [  5.68899703,  16.74926529]], dtype=np.double)
        pos2 = at.atJ2000toGal(pos)
        assert_array_almost_equal(pos2, [
            [358.936378, -85.873779],
            [113.100141, -45.574235]])
        pos3 = at.atGaltoJ2000(pos2)
        assert_array_almost_equal(pos3, pos)

    def test_atGaltoB1950(self):
        """\
        test for atGaltoB1950 and reverse
        """
        pos = np.array([ 6.35801955, 41.75090981], dtype=np.double)
        pos2 = at.atGaltoB1950(pos)
        assert_array_almost_equal(pos2, [234.075549,   0.540651])
        pos3 = at.atB1950toGal(pos2)
        assert_array_almost_equal(pos3, pos)
        pos = np.array([[  8.93388231, -29.38051172],
            [  5.68899703,  16.74926529]], dtype=np.double)
        pos2 = at.atGaltoB1950(pos)
        assert_array_almost_equal(pos2, [
            [301.126286, -32.769103],
            [253.954162, -14.918432]])
        pos3 = at.atB1950toGal(pos2)
        assert_array_almost_equal(pos3, pos)

    def test_atJ2000ToEcliptic(self):
        """\
        test for atJ2000ToEcliptic and reverse
        """
        pos = np.array([ 6.35801955, 41.75090981], dtype=np.double)
        pos2 = at.atJ2000ToEcliptic(pos)
        assert_array_almost_equal(pos2, [24.67735 , 35.315745])
        pos3 = at.atEclipticToJ2000(pos2)
        assert_array_almost_equal(pos3, pos)
        pos = np.array([[  8.93388231, -29.38051172],
            [  5.68899703,  16.74926529]], dtype=np.double)
        pos2 = at.atJ2000ToEcliptic(pos)
        assert_array_almost_equal(pos2, [
            [355.285014, -30.261741],
            [ 11.953215,  13.099652]])
        pos3 = at.atEclipticToJ2000(pos2)
        assert_array_almost_equal(pos3, pos)

    def test_atDegToRA(self):
        """\
        test for atDegToRA
        """
        deg = 78.21657506
        ra = at.atDegToRA(deg)
        assert_array_almost_equal(
            [ra['hour'], ra['min'], ra['sec']],
            [ 5.       , 12.       , 51.9780144]
        )
        deg = np.array([199.51777407, 149.05243946], dtype=np.double)
        ra = at.atDegToRA(deg)
        assert_array_almost_equal(
            [
                [ra[0]['hour'], ra[0]['min'], ra[0]['sec']],
                [ra[1]['hour'], ra[1]['min'], ra[1]['sec']],
            ],
            [
                [13.       , 18.       ,  4.2657768],
                [ 9.       , 56.       , 12.5854704]
            ]
        )

    def test_atDegToDec(self):
        """\
        test for atDegToDec
        """
        deg = -65.33663946
        dec = at.atDegToDec(deg)
        assert_array_almost_equal(
            [dec['sign'], dec['deg'], dec['min'], dec['sec']],
            [-1.      , 65.      , 20.      , 11.902056])
        deg = np.array([  9.02327659, -12.876015  ], dtype=np.double)
        dec = at.atDegToDec(deg)
        assert_array_almost_equal(
            [
                [dec[0]['sign'], dec[0]['deg'], dec[0]['min'], dec[0]['sec']],
                [dec[1]['sign'], dec[1]['deg'], dec[1]['min'], dec[1]['sec']],
            ],
            [
                [ 1.      ,  9.      ,  1.      , 23.795724],
                [-1.      , 12.      , 52.      , 33.654   ]
            ]
        )

    def test_atParseRAToRadian(self):
        """\
        test for atParseRAToRadian
        """
        val = at.atParseRAToRadian("01h20m15.3s")
        assert_almost_equal(val, 0.35017849779701227)
        val = at.atParseRAToRadian("15.3588909")
        #print("15.3588909(deg)={}(rad)".format(val))
        assert_almost_equal(val, 0.2680632156595952)

    def test_atParseDecToRadian(self):
        """\
        test for atParseDecToRadian
        """
        val = at.atParseDecToRadian("+20d15m43.22s")
        assert_almost_equal(val, 0.35363871000182723)
        val = at.atParseDecToRadian("43.75031494")
        assert_almost_equal(val, 0.7635870444874655)

    def test_atQuatToEuler(self):
        """\
        test for atQuatToEuler
        """
        q = np.array([ 1.90931002,  0.11665258, -1.20185093,  0.71761657], dtype=np.double)
        ea = at.atQuatToEuler(q)
        assert_array_almost_equal(
            [ea['phi'], ea['theta'], ea['psi']],
            [2.69939755, 3.14159265, 0.        ]
        )
        q = np.array([
            [ 0.67550666,  0.22370612, -0.14351707, -1.70951295],
            [ 0.81485196,  0.05135934, -0.98748002,  0.13493463]], dtype=np.double)
        ea = at.atQuatToEuler(q)
        assert_array_almost_equal(
            [
                [ea[0]['phi'], ea[0]['theta'], ea[0]['psi']],
                [ea[1]['phi'], ea[1]['theta'], ea[1]['psi']],
            ],
            [
                [0.2352517 , 0.        , 0.        ],
                [3.34034278, 1.23798366, 0.07285857]
            ]
        )

    def test_atEulerToQuat(self):
        """\
        test for atEulerToQuat
        """
        ea = np.array(( 1.2758614 ,  1.06610192, -0.4386156 ), dtype=at.AtEulerAng)
        q = at.atEulerToQuat(ea)
        assert_array_almost_equal(q, [-0.38419081,  0.33260704,  0.35010476,  0.78689049])
        ea = np.array(
            [
                ( 0.07212291,  0.49505272, -0.01493521),
                (-0.50080977, -0.53787328, -0.50798613),
            ],
            dtype=at.AtEulerAng
        )
        q = at.atEulerToQuat(ea)
        assert_array_almost_equal(q, [
            [-1.066153e-02,  2.447744e-01,  2.771857e-02,  9.691251e-01],
            [ 9.534005e-04, -2.657047e-01, -4.659084e-01,  8.439961e-01]])

    def test_atInterpolateQuat(self):
        """\
        test for atInterpolateQuat
        """
        t0=np.array([627782473.386719])
        q0=np.array([[0.351053267429337, 0.812831425977245,
            -0.450039377566186, 0.115956280641032]])
        t1=np.array([627782478.386719])
        q1=np.array([[0.349791221069852, 0.812436132014901,
            -0.451156794809648, 0.118176786184445]])
        q=at.atInterpolateQuat(t0,q0,t1,q1,t0[0])
        assert_array_almost_equal(q, q0)
        q=at.atInterpolateQuat(t0,q0,t1,q1,t1[0])
        assert_array_almost_equal(q-q1, [[ 8.88066459e-09,  2.06265118e-08, -1.14541815e-08,  3.00032801e-09]])

    # def test_atInterpolateEuler(self):
    #     """\
    #     test for atInterpolateEuler
    #     """

    def test_mjd2maxi(self):
        """\
        test for mjd2maxi/maxi2mjd
        """
        at.atMissionTimeInit(os.path.join(os.environ['HEADAS'], 'refdata', 'leapsec.fits'), 0)
        mjd = 59000
        mission = at.optional.mjd2maxi(mjd)
        self.assertEqual(mission, 644198405.0)
        mjd2 = at.optional.maxi2mjd(mission)
        self.assertEqual(mjd2, mjd)
        mjd = np.array([58560, 59000, 59181, ], dtype=np.double)
        mission = at.optional.mjd2maxi(mjd)
        assert_array_almost_equal(mission, [6.06182405e+08, 6.44198405e+08, 6.59836805e+08])
        mjd2 = at.optional.maxi2mjd(mission)
        assert_array_almost_equal(mjd2, mjd)

    def test_gps2maxi(self):
        """\
        test for gps2maxi/maxi2gps
        """
        mission = 644198405.0
        gps = at.optional.maxi2gps(mission)
        self.assertEqual(gps, 1274918418.0)
        mission2 = at.optional.gps2maxi(gps)
        self.assertEqual(mission2, mission)
        mission = np.array([639878405.0, 644198405.0, 651974405.0], dtype=np.double)
        gps = at.optional.maxi2gps(mission)
        assert_array_almost_equal(gps*1.0e-9, np.array([1.27059842e+09, 1.27491842e+09, 1.28269442e+09])*1.0e-9)
        mission2 = at.optional.gps2maxi(gps)
        assert_array_almost_equal(mission2, mission)

    def test_quat2EulerZYX(self):
        """\
        test for atQuat2EulerZYX / Euler2QuatZYX
        """
        ea = np.array([3.4756988 , 3.74176294, 9.54864984])
        q = at.optional.euler2QuatZYX(ea)
        assert_array_almost_equal(q, [ 0.17658384, -0.28109558,  0.10736468,  0.93716394])
        ea2 = at.optional.quat2EulerZYX(q)
        assert_array_almost_equal(ea2, [ 0.334106, -0.60017 ,  0.123872])
        

    def test_quatDiff(self):
        """\
        test for atQuatDiff
        """
        q1 = np.array([0.16434304, 0.25210494, 0.84972172, 0.31435627])
        q2 = np.array([0.48395289, 0.12418355, 0.45406048, 0.02530916])
        q = at.optional.quatDiff(q1, q2)
        assert_array_almost_equal(q, [-0.13883675,  0.33139813, -0.1793498 ,  0.57060852])
        q1 = np.array([[0.14910656, 0.77693336, 0.88214804, 0.62555382],
            [0.41705969, 0.44778623, 0.92694314, 0.18094544]])
        q2 = np.array([[0.83010688, 0.12032086, 0.44163768, 0.09818928],
            [0.75439952, 0.73995392, 0.18753412, 0.64624346]])
        q = at.optional.quatDiff(q1, q2)
        assert_array_almost_equal(q, [[ 0.3220487 , -0.7819198 ,  0.71765539, -0.75995381],
            [-0.60580364,  0.83833948,  0.86447656,  0.82118897]])

    def test_byteswap(self):
        """\
        test for byte swapped array handling
        """
        v = np.array([0.93693726, 0.97315371, 0.47356619], dtype='>d')
        v2 = at.atNorm(v)

if __name__ == '__main__':
    unittest.main()
